<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Traits\HasTenantScope;

class EmployeeLoan extends Model
{
    use HasFactory, SoftDeletes, HasTenantScope;

    protected $fillable = [
        'employee_id',
        'organization_id',
        'loan_number',
        'loan_type',
        'principal_amount',
        'interest_rate',
        'repayment_period_months',
        'monthly_deduction',
        'total_amount',
        'amount_paid',
        'balance',
        'disbursement_date',
        'first_deduction_date',
        'expected_completion_date',
        'actual_completion_date',
        'status',
        'purpose',
        'terms_conditions',
        'approved_by',
        'approved_at',
        'approval_notes',
    ];

    protected $casts = [
        'principal_amount' => 'decimal:2',
        'interest_rate' => 'decimal:2',
        'monthly_deduction' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'amount_paid' => 'decimal:2',
        'balance' => 'decimal:2',
        'disbursement_date' => 'date',
        'first_deduction_date' => 'date',
        'expected_completion_date' => 'date',
        'actual_completion_date' => 'date',
        'approved_at' => 'datetime',
    ];

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function repayments(): HasMany
    {
        return $this->hasMany(LoanRepayment::class, 'loan_id');
    }

    /**
     * Generate unique loan number
     */
    public static function generateLoanNumber(Employee $employee): string
    {
        $year = date('Y');
        $month = date('m');
        $count = static::whereYear('created_at', $year)->whereMonth('created_at', $month)->count() + 1;
        return "LOAN-{$year}{$month}-{$employee->employee_code}-" . str_pad($count, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Calculate total amount with interest
     */
    public function calculateTotalAmount(): float
    {
        $monthlyRate = $this->interest_rate / 12 / 100;
        if ($monthlyRate == 0) {
            return $this->principal_amount;
        }
        
        // Using reducing balance method
        $monthlyPayment = $this->principal_amount * ($monthlyRate * pow(1 + $monthlyRate, $this->repayment_period_months)) / 
                         (pow(1 + $monthlyRate, $this->repayment_period_months) - 1);
        
        return round($monthlyPayment * $this->repayment_period_months, 2);
    }

    /**
     * Calculate monthly deduction
     */
    public function calculateMonthlyDeduction(): float
    {
        return round($this->total_amount / $this->repayment_period_months, 2);
    }

    /**
     * Record a repayment
     */
    public function recordRepayment(float $amount, ?int $payrollId = null, string $method = 'payroll_deduction'): LoanRepayment
    {
        $remainingBalance = $this->balance;
        $interestPortion = $remainingBalance * ($this->interest_rate / 12 / 100);
        $principalPortion = $amount - $interestPortion;

        $repayment = $this->repayments()->create([
            'amount' => $amount,
            'principal_portion' => $principalPortion,
            'interest_portion' => $interestPortion,
            'payment_date' => now(),
            'payment_method' => $method,
            'payroll_id' => $payrollId,
        ]);

        $this->amount_paid += $amount;
        $this->balance -= $amount;

        if ($this->balance <= 0) {
            $this->status = 'completed';
            $this->actual_completion_date = now();
        }

        $this->save();

        return $repayment;
    }
}
