<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use App\Traits\HasTenantScope;

class JobPosting extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'title',
        'department_id',
        'position_id',
        'location',
        'employment_type',
        'description',
        'requirements',
        'responsibilities',
        'salary_range',
        'status',
        'public_token',
        'closes_at',
        'created_by',
        'organization_id',
    ];

    protected $casts = [
        'closes_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($jobPosting) {
            if (empty($jobPosting->public_token)) {
                $jobPosting->public_token = Str::random(32);
            }
        });
    }

    /**
     * Get the department that owns the job posting
     */
    public function department()
    {
        return $this->belongsTo(Department::class);
    }

    /**
     * Get the position that owns the job posting
     */
    public function position()
    {
        return $this->belongsTo(Position::class);
    }

    /**
     * Get the user who created the job posting
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the organization that owns the job posting
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get all applications for this job posting
     */
    public function applications()
    {
        return $this->hasMany(Application::class);
    }

    /**
     * Get the public application URL
     */
    public function getPublicUrlAttribute()
    {
        return url("/apply/{$this->public_token}");
    }

    /**
     * Get applications count
     */
    public function getApplicationsCountAttribute()
    {
        return $this->applications()->count();
    }

    /**
     * Check if job posting is active
     */
    public function isActive()
    {
        return $this->status === 'active' && 
               ($this->closes_at === null || $this->closes_at->isFuture());
    }

    /**
     * Check if job posting is closed
     */
    public function isClosed()
    {
        return $this->status === 'closed' || 
               ($this->closes_at !== null && $this->closes_at->isPast());
    }

    /**
     * Scope to active job postings
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active')
                    ->where(function ($q) {
                        $q->whereNull('closes_at')
                          ->orWhere('closes_at', '>', now());
                    });
    }

    /**
     * Scope to closed job postings
     */
    public function scopeClosed($query)
    {
        return $query->where('status', 'closed')
                    ->orWhere('closes_at', '<=', now());
    }
}