<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;

class LearningPath extends Model
{
    use HasTenantScope;

    protected $fillable = [
        'title',
        'description',
        'image_path',
        'difficulty_level',
        'estimated_duration_hours',
        'target_roles',
        'learning_outcomes',
        'is_active',
        'created_by',
        'organization_id',
    ];

    protected $casts = [
        'target_roles' => 'array',
        'learning_outcomes' => 'array',
        'is_active' => 'boolean',
    ];

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function steps()
    {
        return $this->hasMany(LearningPathStep::class)->orderBy('step_order');
    }

    public function enrollments()
    {
        return $this->hasMany(LearningPathEnrollment::class);
    }

    public function trainingPrograms()
    {
        return $this->belongsToMany(TrainingProgram::class, 'learning_path_steps')
            ->withPivot('step_order', 'is_required', 'prerequisites')
            ->orderBy('learning_path_steps.step_order');
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByDifficulty($query, $level)
    {
        return $query->where('difficulty_level', $level);
    }

    public function getCompletionRateAttribute()
    {
        $totalEnrollments = $this->enrollments()->count();
        if ($totalEnrollments === 0) return 0;

        $completedEnrollments = $this->enrollments()->where('status', 'completed')->count();
        return round(($completedEnrollments / $totalEnrollments) * 100, 2);
    }

    public function getTotalStepsAttribute()
    {
        return $this->steps()->count();
    }

    public function getRequiredStepsAttribute()
    {
        return $this->steps()->where('is_required', true)->count();
    }

    public function getDifficultyColorAttribute()
    {
        return match($this->difficulty_level) {
            'beginner' => 'success',
            'intermediate' => 'warning',
            'advanced' => 'danger',
            default => 'secondary'
        };
    }
}