<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use App\Traits\HasTenantScope;

class LearningPathEnrollment extends Model
{
    use HasTenantScope;

    protected $fillable = [
        'learning_path_id',
        'employee_id',
        'organization_id',
        'status',
        'enrolled_at',
        'started_at',
        'completed_at',
        'progress_percentage',
        'current_step',
        'enrolled_by',
    ];

    protected $casts = [
        'enrolled_at' => 'datetime',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
        'progress_percentage' => 'decimal:2',
    ];

    public function learningPath()
    {
        return $this->belongsTo(LearningPath::class);
    }

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function enrolledBy()
    {
        return $this->belongsTo(User::class, 'enrolled_by');
    }

    public function scopeActive($query)
    {
        return $query->whereIn('status', ['enrolled', 'in_progress']);
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeInProgress($query)
    {
        return $query->where('status', 'in_progress');
    }

    public function getCurrentStepDetails()
    {
        return $this->learningPath->steps()
            ->where('step_order', $this->current_step)
            ->with('trainingProgram')
            ->first();
    }

    public function getNextStepDetails()
    {
        return $this->learningPath->steps()
            ->where('step_order', $this->current_step + 1)
            ->with('trainingProgram')
            ->first();
    }

    public function updateProgress()
    {
        $totalSteps = $this->learningPath->total_steps;
        $completedSteps = $this->getCompletedStepsCount();
        
        $this->progress_percentage = $totalSteps > 0 ? round(($completedSteps / $totalSteps) * 100, 2) : 0;
        
        if ($this->progress_percentage >= 100) {
            $this->status = 'completed';
            $this->completed_at = now();
        } elseif ($this->progress_percentage > 0 && $this->status === 'enrolled') {
            $this->status = 'in_progress';
            $this->started_at = now();
        }
        
        $this->save();
    }

    private function getCompletedStepsCount()
    {
        $stepPrograms = $this->learningPath->steps()->pluck('training_program_id');
        
        return TrainingEnrollment::where('employee_id', $this->employee_id)
            ->where('status', 'completed')
            ->whereHas('session', function($query) use ($stepPrograms) {
                $query->whereIn('training_program_id', $stepPrograms);
            })
            ->count();
    }

    public function canAdvanceToNextStep()
    {
        $currentStep = $this->getCurrentStepDetails();
        if (!$currentStep) return false;

        // Check if current step is completed
        $isCurrentStepCompleted = TrainingEnrollment::where('employee_id', $this->employee_id)
            ->where('status', 'completed')
            ->whereHas('session', function($query) use ($currentStep) {
                $query->where('training_program_id', $currentStep->training_program_id);
            })
            ->exists();

        return $isCurrentStepCompleted;
    }

    public function advanceToNextStep()
    {
        if ($this->canAdvanceToNextStep()) {
            $this->current_step += 1;
            $this->updateProgress();
            return true;
        }
        return false;
    }

    /**
     * Global scope removed - using HasTenantScope trait instead
     */
    protected static function booted()
    {
        // Trait handles global scope
    }
}