<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Traits\HasTenantScope;

class LoanSettings extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'organization_id',
        'max_loan_amount',
        'max_loan_term_months',
        'min_employment_tenure_months',
        'max_outstanding_loans',
        'loan_to_salary_ratio',
        'default_interest_rates',
        'auto_approve_below_amount',
        'require_guarantor_above_amount',
        'require_department_head_approval',
        'require_finance_approval',
        'require_ceo_approval_above_threshold',
        'ceo_approval_threshold',
        'check_credit_score',
        'allow_multiple_loans',
        'allow_loan_with_pending_loan',
        'allow_early_repayment',
        'allow_payment_holidays',
        'max_payment_holidays',
        'early_repayment_penalty_rate',
        'notify_on_approval',
        'notify_before_deduction',
        'notification_days_before',
        'loan_policy_document',
        'terms_and_conditions',
    ];

    protected $casts = [
        'max_loan_amount' => 'decimal:2',
        'loan_to_salary_ratio' => 'decimal:2',
        'default_interest_rates' => 'array',
        'auto_approve_below_amount' => 'decimal:2',
        'require_guarantor_above_amount' => 'decimal:2',
        'require_department_head_approval' => 'boolean',
        'require_finance_approval' => 'boolean',
        'require_ceo_approval_above_threshold' => 'boolean',
        'ceo_approval_threshold' => 'decimal:2',
        'check_credit_score' => 'boolean',
        'allow_multiple_loans' => 'boolean',
        'allow_loan_with_pending_loan' => 'boolean',
        'allow_early_repayment' => 'boolean',
        'allow_payment_holidays' => 'boolean',
        'early_repayment_penalty_rate' => 'decimal:2',
        'notify_on_approval' => 'boolean',
        'notify_before_deduction' => 'boolean',
    ];

    public function organization(): BelongsTo
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get or create loan settings for an organization
     */
    public static function getForOrganization(int $organizationId): self
    {
        return static::firstOrCreate(
            ['organization_id' => $organizationId],
            [
                'default_interest_rates' => static::getDefaultInterestRates(),
            ]
        );
    }

    /**
     * Get default interest rates by loan type
     */
    public static function getDefaultInterestRates(): array
    {
        return [
            'salary_advance' => 0,      // Interest-free
            'emergency' => 5,           // 5% annual
            'personal' => 10,           // 10% annual
            'education' => 8,           // 8% annual
            'housing' => 12,            // 12% annual
            'other' => 10,              // 10% annual
        ];
    }

    /**
     * Get interest rate for a loan type
     */
    public function getInterestRateForType(string $loanType): float
    {
        $rates = $this->default_interest_rates ?? static::getDefaultInterestRates();
        return $rates[$loanType] ?? 10; // Default 10% if not found
    }

    /**
     * Check if employee is eligible for a loan
     */
    public function isEmployeeEligible(Employee $employee, float $loanAmount): array
    {
        $errors = [];

        // Check employment tenure
        if ($employee->hire_date) {
            $tenureMonths = $employee->hire_date->diffInMonths(now());
            if ($tenureMonths < $this->min_employment_tenure_months) {
                $errors[] = "Minimum employment tenure of {$this->min_employment_tenure_months} months required. You have {$tenureMonths} months.";
            }
        }

        // Check maximum loan amount
        if ($loanAmount > $this->max_loan_amount) {
            $errors[] = "Loan amount exceeds maximum of KES " . number_format($this->max_loan_amount, 2);
        }

        // Check loan-to-salary ratio
        if ($employee->basic_salary) {
            $maxAllowed = $employee->basic_salary * $this->loan_to_salary_ratio;
            if ($loanAmount > $maxAllowed) {
                $errors[] = "Loan amount exceeds {$this->loan_to_salary_ratio}x your monthly salary (KES " . number_format($maxAllowed, 2) . ")";
            }
        }

        // Check outstanding loans
        $outstandingLoans = $employee->loans()
            ->whereIn('status', ['approved', 'disbursed', 'active'])
            ->count();
        
        if ($outstandingLoans >= $this->max_outstanding_loans) {
            $errors[] = "You have reached the maximum of {$this->max_outstanding_loans} outstanding loans.";
        }

        // Check pending loans
        if (!$this->allow_loan_with_pending_loan) {
            $pendingLoans = $employee->loans()->where('status', 'pending')->count();
            if ($pendingLoans > 0) {
                $errors[] = "You have a pending loan application. Please wait for approval.";
            }
        }

        return [
            'eligible' => empty($errors),
            'errors' => $errors,
        ];
    }
}
