<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class OrganizationUsage extends Model
{
    use HasFactory;

    protected $fillable = [
        'organization_id',
        'metric_type',
        'metric_value',
        'recorded_date',
        'recorded_at',
    ];

    protected $casts = [
        'metric_value' => 'integer',
        'recorded_date' => 'date',
        'recorded_at' => 'datetime',
    ];

    /**
     * Get the organization that owns the usage record
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Scope for specific metric type
     */
    public function scopeMetricType($query, string $type)
    {
        return $query->where('metric_type', $type);
    }

    /**
     * Scope for date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('recorded_date', [$startDate, $endDate]);
    }

    /**
     * Get latest usage for organization and metric
     */
    public static function getLatestUsage(int $organizationId, string $metricType)
    {
        return self::where('organization_id', $organizationId)
                  ->where('metric_type', $metricType)
                  ->latest('recorded_at')
                  ->first();
    }

    /**
     * Record usage metric
     */
    public static function recordMetric(int $organizationId, string $metricType, int $value)
    {
        return self::create([
            'organization_id' => $organizationId,
            'metric_type' => $metricType,
            'metric_value' => $value,
            'recorded_date' => now()->toDateString(),
            'recorded_at' => now(),
        ]);
    }
}
