<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;

class P9Form extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'employee_id',
        'year',
        'kra_pin',
        'employer_pin',
        'total_gross_pay',
        'total_benefits',
        'total_allowances',
        'total_nssf',
        'total_defined_contribution',
        'total_owner_occupied_interest',
        'total_retirement_contribution',
        'total_insurance_relief',
        'total_paye_tax',
        'total_personal_relief',
        'total_nhif',
        'total_housing_levy',
        'total_helb',
        'monthly_breakdown',
        'status',
        'generated_at',
        'submitted_at',
        'organization_id',
    ];

    protected $casts = [
        'year' => 'integer',
        'total_gross_pay' => 'decimal:2',
        'total_benefits' => 'decimal:2',
        'total_allowances' => 'decimal:2',
        'total_nssf' => 'decimal:2',
        'total_defined_contribution' => 'decimal:2',
        'total_owner_occupied_interest' => 'decimal:2',
        'total_retirement_contribution' => 'decimal:2',
        'total_insurance_relief' => 'decimal:2',
        'total_paye_tax' => 'decimal:2',
        'total_personal_relief' => 'decimal:2',
        'total_nhif' => 'decimal:2',
        'total_housing_levy' => 'decimal:2',
        'total_helb' => 'decimal:2',
        'monthly_breakdown' => 'array',
        'generated_at' => 'datetime',
        'submitted_at' => 'datetime',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    /**
     * Generate P9 form from payroll records
     */
    public static function generateForEmployee($employeeId, $year)
    {
        $employee = Employee::findOrFail($employeeId);
        $payrolls = Payroll::where('employee_id', $employeeId)
            ->where('year', $year)
            ->orderBy('month')
            ->get();

        if ($payrolls->isEmpty()) {
            throw new \Exception('No payroll records found for the specified year.');
        }

        $monthlyBreakdown = [];
        $totals = [
            'total_gross_pay' => 0,
            'total_benefits' => 0,
            'total_allowances' => 0,
            'total_nssf' => 0,
            'total_paye_tax' => 0,
            'total_personal_relief' => 0,
            'total_nhif' => 0,
            'total_housing_levy' => 0,
            'total_helb' => 0,
            'total_insurance_relief' => 0,
        ];

        foreach ($payrolls as $payroll) {
            $monthData = [
                'month' => $payroll->month,
                'basic_salary' => $payroll->basic_salary,
                'allowances' => $payroll->allowances,
                'gross_pay' => $payroll->gross_salary,
                'nssf' => $payroll->nssf_employee,
                'paye_tax' => $payroll->paye_tax,
                'personal_relief' => $payroll->personal_relief,
                'nhif' => $payroll->nhif_deduction,
                'housing_levy' => $payroll->housing_levy,
                'helb' => $payroll->helb_deduction,
                'insurance_relief' => $payroll->insurance_relief,
                'net_pay' => $payroll->net_salary,
            ];

            $monthlyBreakdown[] = $monthData;

            // Add to totals
            $totals['total_gross_pay'] += $payroll->gross_salary;
            $totals['total_allowances'] += $payroll->allowances;
            $totals['total_nssf'] += $payroll->nssf_employee;
            $totals['total_paye_tax'] += $payroll->paye_tax;
            $totals['total_personal_relief'] += $payroll->personal_relief;
            $totals['total_nhif'] += $payroll->nhif_deduction;
            $totals['total_housing_levy'] += $payroll->housing_levy;
            $totals['total_helb'] += $payroll->helb_deduction;
            $totals['total_insurance_relief'] += $payroll->insurance_relief;
        }

        return self::updateOrCreate(
            ['employee_id' => $employeeId, 'year' => $year],
            array_merge($totals, [
                'kra_pin' => $employee->kra_pin ?? '',
                'employer_pin' => config('app.employer_kra_pin', ''),
                'monthly_breakdown' => $monthlyBreakdown,
                'status' => 'generated',
                'generated_at' => now(),
            ])
        );
    }

    /**
     * Get taxable income for the year
     */
    public function getTaxableIncomeAttribute()
    {
        return $this->total_gross_pay - $this->total_nssf - $this->total_defined_contribution;
    }

    /**
     * Get total tax reliefs
     */
    public function getTotalReliefsAttribute()
    {
        return $this->total_personal_relief + $this->total_insurance_relief + $this->total_retirement_contribution;
    }

    /**
     * Get net PAYE after reliefs
     */
    public function getNetPayeAttribute()
    {
        return max(0, $this->total_paye_tax - $this->total_reliefs);
    }
}