<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Traits\HasTenantScope;

class PayrollApprovalWorkflow extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'organization_id',
        'level',
        'approver_role',
        'amount_threshold',
        'required',
        'order',
    ];

    protected $casts = [
        'amount_threshold' => 'decimal:2',
        'required' => 'boolean',
    ];

    public function organization(): BelongsTo
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get workflow for an organization
     */
    public static function getForOrganization(int $organizationId)
    {
        return static::where('organization_id', $organizationId)
            ->orderBy('level')
            ->orderBy('order')
            ->get();
    }

    /**
     * Get default workflow (if none configured)
     */
    public static function getDefaultWorkflow(): array
    {
        return [
            ['level' => 1, 'approver_role' => 'department_head', 'amount_threshold' => 0, 'required' => true, 'order' => 1],
            ['level' => 2, 'approver_role' => 'finance_manager', 'amount_threshold' => 50000, 'required' => true, 'order' => 2],
            ['level' => 3, 'approver_role' => 'ceo', 'amount_threshold' => 200000, 'required' => true, 'order' => 3],
        ];
    }
}
