<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SubscriptionPlan extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'price_monthly',
        'price_annual',
        'max_employees',
        'max_users',
        'max_departments',
        'storage_limit_gb',
        'max_file_size_mb',
        'api_call_limit',
        'features',
        'is_active',
        'sort_order',
    ];

    protected $casts = [
        'price_monthly' => 'decimal:2',
        'price_annual' => 'decimal:2',
        'max_employees' => 'integer',
        'max_users' => 'integer',
        'max_departments' => 'integer',
        'storage_limit_gb' => 'integer',
        'max_file_size_mb' => 'integer',
        'api_call_limit' => 'integer',
        'features' => 'array',
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get all organization subscriptions for this plan
     */
    public function organizationSubscriptions()
    {
        return $this->hasMany(OrganizationSubscription::class);
    }

    /**
     * Get all invoices for this plan
     */
    public function invoices()
    {
        return $this->hasMany(Invoice::class);
    }

    /**
     * Check if plan has unlimited employees
     */
    public function hasUnlimitedEmployees(): bool
    {
        return is_null($this->max_employees);
    }

    /**
     * Check if plan has API access
     */
    public function hasApiAccess(): bool
    {
        return !is_null($this->api_call_limit);
    }

    /**
     * Get feature value
     */
    public function getFeature(string $key, $default = null)
    {
        return data_get($this->features, $key, $default);
    }

    /**
     * Check if plan has specific feature
     */
    public function hasFeature(string $feature): bool
    {
        $features = $this->features ?? [];
        return in_array($feature, $features) || isset($features[$feature]);
    }

    /**
     * Scope for active plans
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for ordered plans
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('price_monthly');
    }
}
