<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use App\Traits\HasTenantScope;

class ThemeSetting extends Model
{
    use HasTenantScope;
    
    protected $fillable = [
        'organization_id',
        'base_theme',
        'primary_color',
        'secondary_color',
        'accent_color',
        'success_color',
        'warning_color',
        'danger_color',
        'info_color',
        'light_color',
        'dark_color',
        'sidebar_bg_color',
        'sidebar_text_color',
        'sidebar_hover_color',
        'sidebar_active_color',
        'primary_font',
        'heading_font',
        'logo_url',
        'sidebar_logo_url',
        'header_logo_url',
        'favicon_url',
        'login_banner_url',
        'sidebar_collapsed_default',
        'dark_mode_enabled',
        'custom_css',
        'is_active',
    ];

    protected $casts = [
        'sidebar_collapsed_default' => 'boolean',
        'dark_mode_enabled' => 'boolean',
        'is_active' => 'boolean',
    ];

    /**
     * Get the organization that owns the theme settings
     */
    public function organization(): BelongsTo
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get the complete theme configuration (base theme + overrides)
     */
    public function getThemeConfig(): array
    {
        // Start with base theme from config
        $baseTheme = config("theme.themes.{$this->base_theme}", config('theme.themes.default'));
        
        // Override with custom colors if set
        $colors = $baseTheme['colors'];
        foreach (['primary', 'secondary', 'accent', 'success', 'warning', 'danger', 'info', 'light', 'dark'] as $colorKey) {
            $dbKey = "{$colorKey}_color";
            if ($this->$dbKey) {
                $colors[$colorKey] = $this->$dbKey;
            }
        }
        
        // Override sidebar colors if set
        $sidebar = $baseTheme['sidebar'];
        if ($this->sidebar_bg_color) $sidebar['background'] = $this->sidebar_bg_color;
        if ($this->sidebar_text_color) $sidebar['text'] = $this->sidebar_text_color;
        if ($this->sidebar_hover_color) $sidebar['hover'] = $this->sidebar_hover_color;
        if ($this->sidebar_active_color) $sidebar['active'] = $this->sidebar_active_color;
        
        return [
            'name' => $baseTheme['name'] . ' (Customized)',
            'colors' => $colors,
            'sidebar' => $sidebar,
            'fonts' => [
                'primary' => $this->primary_font,
                'heading' => $this->heading_font,
            ],
            'logos' => [
                'main' => $this->logo_url,
                'sidebar' => $this->sidebar_logo_url,
                'header' => $this->header_logo_url,
                'favicon' => $this->favicon_url,
                'login_banner' => $this->login_banner_url,
            ],
            'layout' => [
                'sidebar_collapsed' => $this->sidebar_collapsed_default,
                'dark_mode' => $this->dark_mode_enabled,
            ],
            'custom_css' => $this->custom_css,
        ];
    }

    /**
     * Get theme settings for an organization (with caching)
     */
    public static function getForOrganization(int $organizationId): ?self
    {
        return Cache::remember(
            "theme_settings.org.{$organizationId}",
            now()->addHours(24),
            fn() => self::where('organization_id', $organizationId)
                ->where('is_active', true)
                ->first()
        );
    }

    /**
     * Clear cache when theme is updated
     */
    protected static function booted(): void
    {
        static::saved(function ($themeSetting) {
            Cache::forget("theme_settings.org.{$themeSetting->organization_id}");
        });

        static::deleted(function ($themeSetting) {
            Cache::forget("theme_settings.org.{$themeSetting->organization_id}");
        });
    }
}

