<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use App\Traits\HasTenantScope;

class TrainingRecommendation extends Model
{
    use HasTenantScope;

    protected $fillable = [
        'employee_id',
        'organization_id',
        'training_program_id',
        'recommendation_type',
        'reason',
        'priority_score',
        'supporting_data',
        'status',
        'recommended_at',
        'responded_at',
        'recommended_by',
    ];

    protected $casts = [
        'supporting_data' => 'array',
        'recommended_at' => 'datetime',
        'responded_at' => 'datetime',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function trainingProgram()
    {
        return $this->belongsTo(TrainingProgram::class);
    }

    public function recommendedBy()
    {
        return $this->belongsTo(User::class, 'recommended_by');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeAccepted($query)
    {
        return $query->where('status', 'accepted');
    }

    public function scopeHighPriority($query)
    {
        return $query->where('priority_score', '>=', 80);
    }

    public function scopeMediumPriority($query)
    {
        return $query->whereBetween('priority_score', [50, 79]);
    }

    public function scopeLowPriority($query)
    {
        return $query->where('priority_score', '<', 50);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('recommendation_type', $type);
    }

    public function getPriorityLevelAttribute()
    {
        if ($this->priority_score >= 80) return 'High';
        if ($this->priority_score >= 50) return 'Medium';
        return 'Low';
    }

    public function getPriorityColorAttribute()
    {
        if ($this->priority_score >= 80) return 'danger';
        if ($this->priority_score >= 50) return 'warning';
        return 'info';
    }

    public function getRecommendationTypeDisplayAttribute()
    {
        return match($this->recommendation_type) {
            'skill_gap' => 'Skill Gap',
            'career_development' => 'Career Development',
            'performance_improvement' => 'Performance Improvement',
            'mandatory' => 'Mandatory Training',
            'peer_suggestion' => 'Peer Suggestion',
            default => ucfirst(str_replace('_', ' ', $this->recommendation_type))
        };
    }

    public function accept()
    {
        $this->update([
            'status' => 'accepted',
            'responded_at' => now()
        ]);
    }

    public function decline()
    {
        $this->update([
            'status' => 'declined',
            'responded_at' => now()
        ]);
    }

    public function markCompleted()
    {
        $this->update([
            'status' => 'completed',
            'responded_at' => now()
        ]);
    }

    public static function createForEmployee($employeeId, $trainingProgramId, $type, $reason, $priority = 50, $supportingData = [])
    {
        return static::create([
            'employee_id' => $employeeId,
            'training_program_id' => $trainingProgramId,
            'recommendation_type' => $type,
            'reason' => $reason,
            'priority_score' => $priority,
            'supporting_data' => $supportingData,
            'status' => 'pending',
            'recommended_at' => now(),
            'recommended_by' => auth()->id(),
        ]);
    }

    /**
     * Global scope removed - using HasTenantScope trait instead
     */
    protected static function booted()
    {
        // Trait handles global scope
    }
}