<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;
use Carbon\Carbon;

class TrainingSession extends Model
{
    use HasTenantScope;

    protected $fillable = [
        'training_program_id',
        'session_code',
        'start_datetime',
        'end_datetime',
        'location',
        'virtual_meeting_link',
        'session_notes',
        'max_participants',
        'status',
        'instructor_id',
        'actual_cost',
        'created_by',
        'organization_id',
    ];

    protected $casts = [
        'start_datetime' => 'datetime',
        'end_datetime' => 'datetime',
        'actual_cost' => 'decimal:2',
    ];

    public function program()
    {
        return $this->belongsTo(TrainingProgram::class, 'training_program_id');
    }

    public function instructor()
    {
        return $this->belongsTo(User::class, 'instructor_id');
    }

    public function enrollments()
    {
        return $this->hasMany(TrainingEnrollment::class);
    }

    public function attendees()
    {
        return $this->belongsToMany(Employee::class, 'training_enrollments')
            ->withPivot(['status', 'completion_score', 'rating', 'feedback'])
            ->withTimestamps();
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function scopeUpcoming($query)
    {
        return $query->where('start_datetime', '>', now())
            ->where('status', 'scheduled');
    }

    public function scopeInProgress($query)
    {
        return $query->where('status', 'in_progress');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function getAvailableSpotsAttribute()
    {
        $maxParticipants = $this->max_participants ?? $this->program->max_participants;
        if (!$maxParticipants) return null;
        
        $enrolled = $this->enrollments()->whereIn('status', ['enrolled', 'attended', 'completed'])->count();
        return max(0, $maxParticipants - $enrolled);
    }

    public function getIsFullAttribute()
    {
        return $this->available_spots === 0;
    }

    public function getDurationInHoursAttribute()
    {
        return $this->start_datetime->diffInHours($this->end_datetime, true);
    }
}