<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasRoles;
use Spatie\OneTimePasswords\Models\Concerns\HasOneTimePasswords;
use App\Traits\Auditable;

class User extends Authenticatable implements MustVerifyEmail
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, HasRoles, HasOneTimePasswords, Auditable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'photo',
        'phone',
        'organization_id',
        'is_system_admin',
        'last_login_at',
        'locked_until',
        'two_factor_secret',
        'two_factor_recovery_codes',
        'two_factor_confirmed_at',
        'last_otp_sent_at',
        'must_verify_email',
        'first_login_completed',
        'initial_password',
        'email_verified_at',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_secret',
        'two_factor_recovery_codes',
    ];

    /**
     * Attributes to exclude from auditing
     */
    protected $auditExcluded = [
        'password',
        'remember_token',
        'email_verified_at',
        'updated_at',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_system_admin' => 'boolean',
            'last_login_at' => 'datetime',
            'locked_until' => 'datetime',
            'two_factor_recovery_codes' => 'encrypted:array',
            'two_factor_confirmed_at' => 'datetime',
            'last_otp_sent_at' => 'datetime',
            'must_verify_email' => 'boolean',
            'first_login_completed' => 'boolean',
        ];
    }

    /**
     * Get the employee record for this user
     */
    public function employee()
    {
        return $this->hasOne(Employee::class);
    }

    /**
     * Get the organization this user belongs to
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Check if user is a system admin
     */
    public function isSystemAdmin()
    {
        return $this->is_system_admin;
    }

    /**
     * Check if user is an organization admin
     */
    public function isOrganizationAdmin()
    {
        return $this->hasRole('org_admin');
    }

    /**
     * Check if user can manage roles (has manage_roles permission)
     */
    public function canManageRoles()
    {
        return $this->can('manage_roles');
    }

    /**
     * Get the primary role for this user (for backward compatibility)
     */
    public function getPrimaryRoleAttribute()
    {
        return $this->roles->first();
    }

    /**
     * Get the primary role name for this user
     */
    public function getPrimaryRoleNameAttribute()
    {
        $role = $this->roles->first();
        return $role ? $role->name : null;
    }

    /**
     * Send the email verification notification (override to prevent automatic sending)
     */
    public function sendEmailVerificationNotification()
    {
        // Do nothing - we handle OTP sending manually in the verification controller
        // This prevents Laravel from automatically sending verification link emails
    }

    /**
     * Determine if the user has verified their email address.
     *
     * @return bool
     */
    public function hasVerifiedEmail()
    {
        return !is_null($this->email_verified_at);
    }

    /**
     * Mark the given user's email as verified.
     *
     * @return bool
     */
    public function markEmailAsVerified()
    {
        return $this->forceFill([
            'email_verified_at' => $this->freshTimestamp(),
        ])->save();
    }
}
