<?php

namespace App\Policies;

use App\Models\Announcement;
use App\Models\User;
use Illuminate\Auth\Access\Response;

class AnnouncementPolicy
{
    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        return $user->hasPermissionTo('view_announcements');
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Announcement $announcement): bool
    {
        // Users can view published announcements or their own drafts
        if ($announcement->canBeViewed()) {
            return $user->hasPermissionTo('view_announcements');
        }

        // Allow viewing own announcements or if user can manage announcements
        return $user->id === $announcement->created_by || 
               $user->hasPermissionTo('manage_announcements');
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        return $user->hasPermissionTo('create_announcements');
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Announcement $announcement): bool
    {
        // Users can update their own announcements or if they have manage permission
        return $user->id === $announcement->created_by || 
               $user->hasPermissionTo('manage_announcements');
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Announcement $announcement): bool
    {
        // Users can delete their own announcements or if they have manage permission
        return $user->id === $announcement->created_by || 
               $user->hasPermissionTo('manage_announcements');
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Announcement $announcement): bool
    {
        return $user->hasPermissionTo('manage_announcements');
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Announcement $announcement): bool
    {
        return $user->hasPermissionTo('manage_announcements');
    }
}