<?php

namespace App\Policies;

use App\Models\Attendance;
use App\Models\User;

class AttendancePolicy
{
    /**
     * Determine whether the user can view any attendance records.
     */
    public function viewAny(User $user): bool
    {
        return $user->can('view_attendance');
    }

    /**
     * Determine whether the user can view the attendance record.
     */
    public function view(User $user, Attendance $attendance): bool
    {
        // Users can view their own attendance
        if ($user->employee && $user->employee->id === $attendance->employee_id) {
            return true;
        }

        return $user->can('view_attendance');
    }

    /**
     * Determine whether the user can create attendance records.
     */
    public function create(User $user): bool
    {
        return $user->can('create_attendance');
    }

    /**
     * Determine whether the user can update the attendance record.
     */
    public function update(User $user, Attendance $attendance): bool
    {
        return $user->can('edit_attendance');
    }

    /**
     * Determine whether the user can delete the attendance record.
     */
    public function delete(User $user, Attendance $attendance): bool
    {
        return $user->can('delete_attendance');
    }

    /**
     * Determine whether the user can restore the attendance record.
     */
    public function restore(User $user, Attendance $attendance): bool
    {
        return $user->can('delete_attendance');
    }

    /**
     * Determine whether the user can permanently delete the attendance record.
     */
    public function forceDelete(User $user, Attendance $attendance): bool
    {
        return $user->can('delete_attendance');
    }

    /**
     * Determine whether the user can check in/out.
     */
    public function checkInOut(User $user): bool
    {
        // All authenticated users can check in/out for themselves
        return $user->employee !== null;
    }
}
