<?php

namespace App\Policies;

use App\Models\EmployeeLoan;
use App\Models\User;

class EmployeeLoanPolicy
{
    /**
     * Determine whether the user can view any loans.
     */
    public function viewAny(User $user): bool
    {
        return $user->can('view_payroll');
    }

    /**
     * Determine whether the user can view the loan.
     */
    public function view(User $user, EmployeeLoan $loan): bool
    {
        // Users can view their own loans
        if ($user->employee && $user->employee->id === $loan->employee_id) {
            return true;
        }

        return $user->can('view_payroll');
    }

    /**
     * Determine whether the user can create loans.
     */
    public function create(User $user): bool
    {
        return $user->can('manage_payroll');
    }

    /**
     * Determine whether the user can update the loan.
     */
    public function update(User $user, EmployeeLoan $loan): bool
    {
        // Only pending loans can be edited
        if ($loan->status !== 'pending') {
            return false;
        }

        return $user->can('manage_payroll');
    }

    /**
     * Determine whether the user can delete the loan.
     */
    public function delete(User $user, EmployeeLoan $loan): bool
    {
        // Only pending or cancelled loans can be deleted
        if (!in_array($loan->status, ['pending', 'cancelled'])) {
            return false;
        }

        return $user->can('manage_payroll');
    }

    /**
     * Determine whether the user can approve the loan.
     */
    public function approve(User $user, EmployeeLoan $loan): bool
    {
        // Only pending loans can be approved
        if ($loan->status !== 'pending') {
            return false;
        }

        return $user->can('manage_payroll');
    }

    /**
     * Determine whether the user can reject the loan.
     */
    public function reject(User $user, EmployeeLoan $loan): bool
    {
        return $this->approve($user, $loan);
    }

    /**
     * Determine whether the user can disburse the loan.
     */
    public function disburse(User $user, EmployeeLoan $loan): bool
    {
        // Only approved loans can be disbursed
        if ($loan->status !== 'approved') {
            return false;
        }

        return $user->can('manage_payroll');
    }

    /**
     * Determine whether the user can activate the loan.
     */
    public function activate(User $user, EmployeeLoan $loan): bool
    {
        // Only disbursed loans can be activated
        if ($loan->status !== 'disbursed') {
            return false;
        }

        return $user->can('manage_payroll');
    }

    /**
     * Determine whether the user can view repayment history.
     */
    public function viewRepayments(User $user, EmployeeLoan $loan): bool
    {
        return $this->view($user, $loan);
    }
}
