<?php

namespace App\Policies;

use App\Models\Invoice;
use App\Models\User;

class InvoicePolicy
{
    /**
     * Determine if the user can view the invoice
     */
    public function view(User $user, Invoice $invoice): bool
    {
        // System admin can view all invoices
        if ($user->is_system_admin) {
            return true;
        }

        // User can only view invoices for their organization
        return $user->organization_id === $invoice->organization_id;
    }

    /**
     * Determine if the user can pay the invoice
     */
    public function pay(User $user, Invoice $invoice): bool
    {
        // System admin cannot pay invoices
        if ($user->is_system_admin) {
            return false;
        }

        // User must be from the same organization
        if ($user->organization_id !== $invoice->organization_id) {
            return false;
        }

        // Invoice must not be already paid
        if ($invoice->isPaid()) {
            return false;
        }

        // User must have permission (org admin or accountant)
        return $user->hasAnyRole(['org_admin', 'accountant']);
    }

    /**
     * Determine if the user can view any invoices
     */
    public function viewAny(User $user): bool
    {
        // System admin can view all
        if ($user->is_system_admin) {
            return true;
        }

        // Organization users can view their org's invoices
        return $user->hasAnyRole(['org_admin', 'accountant']);
    }
}
