<?php

namespace App\Policies;

use App\Models\LeaveRequest;
use App\Models\User;

class LeaveRequestPolicy
{
    /**
     * Determine whether the user can view any leave requests.
     */
    public function viewAny(User $user): bool
    {
        return $user->can('view_leaves');
    }

    /**
     * Determine whether the user can view the leave request.
     */
    public function view(User $user, LeaveRequest $leaveRequest): bool
    {
        // Users can view their own leave requests
        if ($user->employee && $user->employee->id === $leaveRequest->employee_id) {
            return true;
        }

        return $user->can('view_leaves');
    }

    /**
     * Determine whether the user can create leave requests.
     */
    public function create(User $user): bool
    {
        return $user->can('create_leaves');
    }

    /**
     * Determine whether the user can update the leave request.
     */
    public function update(User $user, LeaveRequest $leaveRequest): bool
    {
        // Users can edit their own pending leave requests
        if ($user->employee && 
            $user->employee->id === $leaveRequest->employee_id && 
            $leaveRequest->status === 'pending') {
            return true;
        }

        return $user->can('edit_leaves');
    }

    /**
     * Determine whether the user can delete the leave request.
     */
    public function delete(User $user, LeaveRequest $leaveRequest): bool
    {
        // Users can delete their own pending leave requests
        if ($user->employee && 
            $user->employee->id === $leaveRequest->employee_id && 
            $leaveRequest->status === 'pending') {
            return true;
        }

        return $user->can('delete_leaves');
    }

    /**
     * Determine whether the user can restore the leave request.
     */
    public function restore(User $user, LeaveRequest $leaveRequest): bool
    {
        return $user->can('delete_leaves');
    }

    /**
     * Determine whether the user can permanently delete the leave request.
     */
    public function forceDelete(User $user, LeaveRequest $leaveRequest): bool
    {
        return $user->can('delete_leaves');
    }

    /**
     * Determine whether the user can approve leave requests.
     */
    public function approve(User $user, LeaveRequest $leaveRequest): bool
    {
        return $user->can('approve_leaves');
    }

    /**
     * Determine whether the user can reject leave requests.
     */
    public function reject(User $user, LeaveRequest $leaveRequest): bool
    {
        return $user->can('approve_leaves');
    }

    /**
     * Determine whether the user can cancel leave requests.
     */
    public function cancel(User $user, LeaveRequest $leaveRequest): bool
    {
        // Users can cancel their own approved leave requests
        if ($user->employee && 
            $user->employee->id === $leaveRequest->employee_id && 
            $leaveRequest->status === 'approved') {
            return true;
        }

        return $user->can('approve_leaves');
    }
}
