<?php

namespace App\Policies;

use App\Models\User;
use Spatie\Permission\Models\Permission;

class PermissionPolicy
{
    /**
     * Determine whether the user can view any permissions.
     */
    public function viewAny(User $user): bool
    {
        return $user->can('view_permissions');
    }

    /**
     * Determine whether the user can view the permission.
     */
    public function view(User $user, Permission $permission): bool
    {
        // Org admins cannot view system admin permissions
        if (!$user->isSystemAdmin() && $this->isSystemAdminPermission($permission)) {
            return false;
        }

        return $user->can('view_permissions');
    }

    /**
     * Determine whether the user can create permissions.
     */
    public function create(User $user): bool
    {
        return $user->can('create_permissions');
    }

    /**
     * Determine whether the user can update the permission.
     */
    public function update(User $user, Permission $permission): bool
    {
        // Org admins cannot update system admin permissions
        if (!$user->isSystemAdmin() && $this->isSystemAdminPermission($permission)) {
            return false;
        }

        return $user->can('edit_permissions');
    }

    /**
     * Determine whether the user can delete the permission.
     */
    public function delete(User $user, Permission $permission): bool
    {
        // Org admins cannot delete system admin permissions
        if (!$user->isSystemAdmin() && $this->isSystemAdminPermission($permission)) {
            return false;
        }

        return $user->can('delete_permissions');
    }

    /**
     * Determine whether the user can restore the permission.
     */
    public function restore(User $user, Permission $permission): bool
    {
        return $user->can('delete_permissions');
    }

    /**
     * Determine whether the user can permanently delete the permission.
     */
    public function forceDelete(User $user, Permission $permission): bool
    {
        return $user->can('delete_permissions');
    }

    /**
     * Check if a permission is related to system admin functionality
     */
    private function isSystemAdminPermission(Permission $permission): bool
    {
        $systemAdminPermissions = [
            'manage_system_admin',
            'view_system_admin',
            'create_system_admin',
            'edit_system_admin',
            'delete_system_admin'
        ];

        return in_array($permission->name, $systemAdminPermissions);
    }
}
