<?php

namespace App\Policies;

use App\Models\User;
use Spatie\Permission\Models\Role;

class RolePolicy
{
    /**
     * Determine whether the user can view any roles.
     */
    public function viewAny(User $user): bool
    {
        return $user->can('view_roles');
    }

    /**
     * Determine whether the user can view the role.
     */
    public function view(User $user, Role $role): bool
    {
        // Org admins cannot view system admin roles
        if (!$user->isSystemAdmin() && $role->name === 'system_admin') {
            return false;
        }

        return $user->can('view_roles');
    }

    /**
     * Determine whether the user can create roles.
     */
    public function create(User $user): bool
    {
        return $user->can('create_roles');
    }

    /**
     * Determine whether the user can update the role.
     */
    public function update(User $user, Role $role): bool
    {
        // Prevent editing system admin role unless user is system admin
        if ($role->name === 'system_admin' && !$user->isSystemAdmin()) {
            return false;
        }

        // Only system admins can edit the org_admin role
        if ($role->name === 'org_admin' && !$user->isSystemAdmin()) {
            return false;
        }

        return $user->can('edit_roles');
    }

    /**
     * Determine whether the user can delete the role.
     */
    public function delete(User $user, Role $role): bool
    {
        // Prevent deleting system admin role
        if ($role->name === 'system_admin') {
            return false;
        }

        // Prevent deleting org_admin role
        if ($role->name === 'org_admin') {
            return false;
        }

        return $user->can('delete_roles');
    }

    /**
     * Determine whether the user can restore the role.
     */
    public function restore(User $user, Role $role): bool
    {
        return $user->can('delete_roles');
    }

    /**
     * Determine whether the user can permanently delete the role.
     */
    public function forceDelete(User $user, Role $role): bool
    {
        return $user->can('delete_roles');
    }
}
