<?php

namespace App\Policies;

use App\Models\User;
use App\Models\TrainingRecommendation;
use App\Models\Employee;

class TrainingRecommendationPolicy
{
    public function viewAny(User $user)
    {
        return $user->can('view_training_recommendations') || 
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager', 'Manager']);
    }

    public function view(User $user, TrainingRecommendation $recommendation)
    {
        // Users can view their own recommendations
        if ($user->employee && $user->employee->id === $recommendation->employee_id) {
            return true;
        }

        // Managers can view recommendations for their team members
        if ($user->can('view_team_training_recommendations')) {
            $employee = Employee::find($recommendation->employee_id);
            if ($employee && $employee->manager_id === $user->employee?->id) {
                return true;
            }
        }

        // HR and Training managers can view all
        return $user->can('view_all_training_recommendations') ||
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager']);
    }

    public function create(User $user)
    {
        return $user->can('create_training_recommendations') ||
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager', 'Manager']);
    }

    public function update(User $user, TrainingRecommendation $recommendation)
    {
        // Only the recommender or authorized users can update
        if ($recommendation->recommended_by === $user->id) {
            return true;
        }

        return $user->can('manage_training_recommendations') ||
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager']);
    }

    public function delete(User $user, TrainingRecommendation $recommendation)
    {
        return $user->can('delete_training_recommendations') ||
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager']);
    }

    public function respond(User $user, TrainingRecommendation $recommendation)
    {
        // Employees can respond to their own recommendations
        if ($user->employee && $user->employee->id === $recommendation->employee_id) {
            return true;
        }

        // Managers can respond on behalf of their team members
        if ($user->can('respond_team_training_recommendations')) {
            $employee = Employee::find($recommendation->employee_id);
            if ($employee && $employee->manager_id === $user->employee?->id) {
                return true;
            }
        }

        return false;
    }
}