<?php

namespace App\Providers;

use App\Listeners\AuditAuthenticationEvents;
use App\Services\AuditService;
use App\Traits\Auditable;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Event;
use Illuminate\Console\Events\CommandFinished;
use Illuminate\Console\Events\CommandStarting;

class AuditServiceProvider extends ServiceProvider
{
    /**
     * Register services
     */
    public function register(): void
    {
        // Merge audit configuration
        $this->mergeConfigFrom(
            __DIR__.'/../../config/audit.php', 'audit'
        );

        // Register audit service as singleton
        $this->app->singleton(AuditService::class, function ($app) {
            return new AuditService();
        });
    }

    /**
     * Bootstrap services
     */
    public function boot(): void
    {
        // Publish configuration
        if ($this->app->runningInConsole()) {
            $this->publishes([
                __DIR__.'/../../config/audit.php' => config_path('audit.php'),
            ], 'audit-config');
        }

        // Register event listeners
        $this->registerEventListeners();

        // Auto-apply Auditable trait to configured models
        $this->autoApplyAuditableTraits();

        // Register console command auditing
        $this->registerConsoleAuditing();
    }

    /**
     * Register authentication and other event listeners
     */
    private function registerEventListeners(): void
    {
        // Register authentication event subscriber
        Event::subscribe(AuditAuthenticationEvents::class);

        // Register model event listeners for auto-audit models
        $autoAuditModels = config('audit.auto_audit_models', []);
        
        foreach ($autoAuditModels as $modelClass) {
            if (class_exists($modelClass)) {
                // The Auditable trait will handle the actual event registration
                // This just ensures the models are loaded
                $this->app->make($modelClass);
            }
        }
    }

    /**
     * Automatically apply Auditable trait to configured models
     */
    private function autoApplyAuditableTraits(): void
    {
        $autoAuditModels = config('audit.auto_audit_models', []);
        
        foreach ($autoAuditModels as $modelClass) {
            if (class_exists($modelClass)) {
                // Check if model already uses Auditable trait
                $traits = class_uses_recursive($modelClass);
                if (!in_array(Auditable::class, $traits)) {
                    // Log warning that model should use Auditable trait
                    if (config('app.debug')) {
                        logger()->warning("Model {$modelClass} is configured for auto-auditing but doesn't use the Auditable trait");
                    }
                }
            }
        }
    }

    /**
     * Register console command auditing
     */
    private function registerConsoleAuditing(): void
    {
        if (!config('audit.console_enabled', false)) {
            return;
        }

        Event::listen(CommandStarting::class, function (CommandStarting $event) {
            if ($this->shouldAuditCommand($event->command)) {
                AuditService::logSystemAdmin(
                    'command_started',
                    "Console command started: {$event->command}",
                    [
                        'command' => $event->command,
                        'arguments' => $event->input->getArguments(),
                        'options' => $event->input->getOptions(),
                    ]
                );
            }
        });

        Event::listen(CommandFinished::class, function (CommandFinished $event) {
            if ($this->shouldAuditCommand($event->command)) {
                AuditService::logSystemAdmin(
                    'command_finished',
                    "Console command finished: {$event->command} (Exit code: {$event->exitCode})",
                    [
                        'command' => $event->command,
                        'exit_code' => $event->exitCode,
                        'arguments' => $event->input->getArguments(),
                        'options' => $event->input->getOptions(),
                    ]
                );
            }
        });
    }

    /**
     * Determine if a console command should be audited
     */
    private function shouldAuditCommand(string $command): bool
    {
        // Skip routine commands
        $skipCommands = [
            'schedule:run',
            'queue:work',
            'queue:listen',
            'horizon:work',
            'telescope:prune',
            'audit:cleanup', // Our own cleanup command
        ];

        foreach ($skipCommands as $skipCommand) {
            if (str_contains($command, $skipCommand)) {
                return false;
            }
        }

        // Only audit important commands
        $auditCommands = [
            'migrate',
            'db:seed',
            'cache:clear',
            'config:clear',
            'route:clear',
            'view:clear',
            'optimize',
            'down',
            'up',
            'user:create',
            'role:assign',
        ];

        foreach ($auditCommands as $auditCommand) {
            if (str_contains($command, $auditCommand)) {
                return true;
            }
        }

        return false;
    }
}