<?php

namespace App\Services;

use Illuminate\Support\Str;

class InputSanitizationService
{
    /**
     * Sanitize string input to prevent XSS and other attacks
     */
    public function sanitizeString(string $input, array $options = []): string
    {
        $options = array_merge([
            'strip_tags' => true,
            'trim' => true,
            'remove_special_chars' => false,
            'max_length' => null,
            'allowed_tags' => '',
        ], $options);

        // Trim whitespace
        if ($options['trim']) {
            $input = trim($input);
        }

        // Strip HTML tags
        if ($options['strip_tags']) {
            $input = strip_tags($input, $options['allowed_tags']);
        }

        // Remove special characters if requested
        if ($options['remove_special_chars']) {
            $input = preg_replace('/[^a-zA-Z0-9\s\-_.]/', '', $input);
        }

        // Limit length
        if ($options['max_length']) {
            $input = Str::limit($input, $options['max_length'], '');
        }

        // HTML entity encoding for output safety
        $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');

        return $input;
    }

    /**
     * Sanitize email input
     */
    public function sanitizeEmail(string $email): string
    {
        $email = trim(strtolower($email));
        $email = filter_var($email, FILTER_SANITIZE_EMAIL);
        return $email;
    }

    /**
     * Sanitize phone number
     */
    public function sanitizePhone(string $phone): string
    {
        // Remove all non-numeric characters except + and -
        $phone = preg_replace('/[^0-9+\-\s()]/', '', $phone);
        $phone = trim($phone);
        return $phone;
    }

    /**
     * Sanitize URL input
     */
    public function sanitizeUrl(string $url): string
    {
        $url = trim($url);
        $url = filter_var($url, FILTER_SANITIZE_URL);
        
        // Validate URL format
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            return '';
        }

        return $url;
    }

    /**
     * Sanitize file name for safe storage
     */
    public function sanitizeFileName(string $filename): string
    {
        // Remove path information
        $filename = basename($filename);
        
        // Remove dangerous characters
        $filename = preg_replace('/[^a-zA-Z0-9\-_\.]/', '_', $filename);
        
        // Remove multiple dots (directory traversal protection)
        $filename = preg_replace('/\.{2,}/', '.', $filename);
        
        // Ensure filename is not empty and has reasonable length
        if (empty($filename) || $filename === '.') {
            $filename = 'file_' . time();
        }
        
        $filename = Str::limit($filename, 100, '');
        
        return $filename;
    }

    /**
     * Sanitize array of inputs
     */
    public function sanitizeArray(array $data, array $rules = []): array
    {
        $sanitized = [];
        
        foreach ($data as $key => $value) {
            $rule = $rules[$key] ?? ['type' => 'string'];
            
            if (is_array($value)) {
                $sanitized[$key] = $this->sanitizeArray($value, $rule['nested'] ?? []);
            } else {
                $sanitized[$key] = $this->sanitizeByType($value, $rule);
            }
        }
        
        return $sanitized;
    }

    /**
     * Sanitize input based on type
     */
    private function sanitizeByType($value, array $rule)
    {
        $type = $rule['type'] ?? 'string';
        
        switch ($type) {
            case 'email':
                return $this->sanitizeEmail($value);
            
            case 'phone':
                return $this->sanitizePhone($value);
            
            case 'url':
                return $this->sanitizeUrl($value);
            
            case 'filename':
                return $this->sanitizeFileName($value);
            
            case 'integer':
                return (int) filter_var($value, FILTER_SANITIZE_NUMBER_INT);
            
            case 'float':
                return (float) filter_var($value, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
            
            case 'boolean':
                return filter_var($value, FILTER_VALIDATE_BOOLEAN);
            
            case 'html':
                // For rich text content - allow specific HTML tags
                $allowedTags = $rule['allowed_tags'] ?? '<p><br><strong><em><ul><ol><li><a>';
                return strip_tags($value, $allowedTags);
            
            case 'string':
            default:
                $options = $rule['options'] ?? [];
                return $this->sanitizeString($value, $options);
        }
    }

    /**
     * Detect and prevent SQL injection patterns
     */
    public function detectSqlInjection(string $input): bool
    {
        $patterns = [
            '/(\b(SELECT|INSERT|UPDATE|DELETE|DROP|CREATE|ALTER|EXEC|UNION|SCRIPT)\b)/i',
            '/(\b(OR|AND)\s+\d+\s*=\s*\d+)/i',
            '/(\b(OR|AND)\s+[\'"]?\w+[\'"]?\s*=\s*[\'"]?\w+[\'"]?)/i',
            '/(\-\-|\#|\/\*|\*\/)/i',
            '/(;|\||&)/i',
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $input)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Detect XSS patterns
     */
    public function detectXss(string $input): bool
    {
        $patterns = [
            '/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/mi',
            '/<iframe\b[^<]*(?:(?!<\/iframe>)<[^<]*)*<\/iframe>/mi',
            '/javascript:/i',
            '/on\w+\s*=/i',
            '/<object\b[^<]*(?:(?!<\/object>)<[^<]*)*<\/object>/mi',
            '/<embed\b[^<]*(?:(?!<\/embed>)<[^<]*)*<\/embed>/mi',
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $input)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Comprehensive security check
     */
    public function isInputSafe(string $input): array
    {
        return [
            'safe' => !$this->detectSqlInjection($input) && !$this->detectXss($input),
            'sql_injection' => $this->detectSqlInjection($input),
            'xss' => $this->detectXss($input),
        ];
    }
}