<?php

namespace App\Services;

use App\Models\PayrollSettings;
use Illuminate\Support\Facades\Auth;

class KenyaTaxCalculator
{
    protected static ?PayrollSettings $settings = null;

    /**
     * Get payroll settings for the current organization
     */
    protected static function getSettings(): PayrollSettings
    {
        if (self::$settings === null) {
            $organizationId = Auth::user()->organization_id ?? 1;
            self::$settings = PayrollSettings::getForOrganization($organizationId);
        }
        return self::$settings;
    }

    /**
     * Reset cached settings (useful for testing)
     */
    public static function resetSettings(): void
    {
        self::$settings = null;
    }

    /**
     * Calculate PAYE tax using settings from database
     */
    public static function calculatePAYE(float $taxableIncome): float
    {
        $settings = self::getSettings();
        return $settings->calculatePayeTax($taxableIncome);
    }

    /**
     * Calculate SHIF deduction (replaced NHIF in 2024)
     */
    public static function calculateSHIF(float $grossSalary): float
    {
        $settings = self::getSettings();
        return $settings->calculateShifContribution($grossSalary);
    }

    /**
     * Calculate NHIF deduction (deprecated - kept for backward compatibility)
     * @deprecated Use calculateSHIF instead
     */
    public static function calculateNHIF(float $grossSalary): float
    {
        return self::calculateSHIF($grossSalary);
    }

    /**
     * Calculate NSSF employee contribution
     */
    public static function calculateNSSFEmployee(float $basicSalary): float
    {
        $settings = self::getSettings();
        return $settings->calculateNssfContribution($basicSalary, true);
    }

    /**
     * Calculate NSSF employer contribution
     */
    public static function calculateNSSFEmployer(float $basicSalary): float
    {
        $settings = self::getSettings();
        return $settings->calculateNssfContribution($basicSalary, false);
    }

    /**
     * Calculate Housing Levy
     */
    public static function calculateHousingLevy(float $grossSalary): float
    {
        $settings = self::getSettings();
        return $settings->calculateHousingLevyContribution($grossSalary, true);
    }

    /**
     * Calculate Housing Levy (Employer)
     */
    public static function calculateHousingLevyEmployer(float $grossSalary): float
    {
        $settings = self::getSettings();
        return $settings->calculateHousingLevyContribution($grossSalary, false);
    }

    /**
     * Calculate taxable income
     */
    public static function calculateTaxableIncome(
        float $grossSalary,
        float $nssfEmployee,
        ?float $personalRelief = null,
        float $insuranceRelief = 0,
        float $pensionRelief = 0
    ): float {
        $settings = self::getSettings();
        $personalRelief = $personalRelief ?? $settings->personal_relief;
        
        $taxableIncome = $grossSalary - $nssfEmployee - $personalRelief - $insuranceRelief - $pensionRelief;
        
        return max(0, $taxableIncome);
    }

    /**
     * Calculate insurance relief using settings
     */
    public static function calculateInsuranceRelief(float $insurancePremium): float
    {
        $settings = self::getSettings();
        $relief = $insurancePremium * $settings->insurance_relief_rate;
        return min($settings->insurance_relief_max, round($relief, 2));
    }

    /**
     * Calculate pension relief using settings
     */
    public static function calculatePensionRelief(float $pensionContribution, float $pensionablePay): float
    {
        $settings = self::getSettings();
        $maxRelief = min($settings->pension_relief_max, $pensionablePay * $settings->pension_relief_rate);
        return min($maxRelief, $pensionContribution);
    }

    /**
     * Calculate complete payroll for an employee
     */
    public static function calculatePayroll(array $salaryComponents): array
    {
        // Extract components
        $basicSalary = $salaryComponents['basic_salary'] ?? 0;
        $houseAllowance = $salaryComponents['house_allowance'] ?? 0;
        $transportAllowance = $salaryComponents['transport_allowance'] ?? 0;
        $medicalAllowance = $salaryComponents['medical_allowance'] ?? 0;
        $otherAllowances = $salaryComponents['other_allowances'] ?? 0;
        $bonuses = $salaryComponents['bonuses'] ?? 0;
        $overtimePay = $salaryComponents['overtime_pay'] ?? 0;

        // Calculate gross salary
        $totalAllowances = $houseAllowance + $transportAllowance + $medicalAllowance + $otherAllowances;
        $grossSalary = $basicSalary + $totalAllowances + $bonuses + $overtimePay;

        // Calculate statutory deductions
        $nssfEmployee = self::calculateNSSFEmployee($basicSalary);
        $nssfEmployer = self::calculateNSSFEmployer($basicSalary);
        $shifDeduction = self::calculateSHIF($grossSalary); // Using SHIF instead of NHIF
        $housingLevy = self::calculateHousingLevy($grossSalary);

        // Calculate tax reliefs
        $settings = self::getSettings();
        $personalRelief = $settings->personal_relief;
        $insuranceRelief = self::calculateInsuranceRelief($salaryComponents['insurance_premium'] ?? 0);
        $pensionRelief = self::calculatePensionRelief($salaryComponents['pension_contribution'] ?? 0, $basicSalary);

        // Calculate taxable income and PAYE
        $taxableIncome = self::calculateTaxableIncome($grossSalary, $nssfEmployee, $personalRelief, $insuranceRelief, $pensionRelief);
        $payeTax = self::calculatePAYE($taxableIncome);

        // Apply reliefs to reduce PAYE
        $finalPAYE = max(0, $payeTax - $personalRelief - $insuranceRelief - $pensionRelief);

        // Other deductions
        $helbDeduction = $salaryComponents['helb_deduction'] ?? 0;
        $loanDeductions = $salaryComponents['loan_deductions'] ?? 0;
        $advanceDeductions = $salaryComponents['advance_deductions'] ?? 0;
        $disciplinaryDeductions = $salaryComponents['disciplinary_deductions'] ?? 0;
        $otherDeductions = $salaryComponents['other_deductions'] ?? 0;

        // Calculate total deductions
        $totalDeductions = $finalPAYE + $shifDeduction + $nssfEmployee + $housingLevy + 
                          $helbDeduction + $loanDeductions + $advanceDeductions + 
                          $disciplinaryDeductions + $otherDeductions;

        // Calculate net salary
        $netSalary = $grossSalary - $totalDeductions;

        return [
            'basic_salary' => $basicSalary,
            'house_allowance' => $houseAllowance,
            'transport_allowance' => $transportAllowance,
            'medical_allowance' => $medicalAllowance,
            'other_allowances' => $otherAllowances,
            'allowances' => $totalAllowances,
            'bonuses' => $bonuses,
            'overtime_pay' => $overtimePay,
            'gross_salary' => $grossSalary,
            'taxable_income' => $taxableIncome,
            'paye_tax' => $finalPAYE,
            'shif_deduction' => $shifDeduction, // Updated from nhif_deduction
            'nhif_deduction' => $shifDeduction, // Kept for backward compatibility
            'nssf_employee' => $nssfEmployee,
            'nssf_employer' => $nssfEmployer,
            'housing_levy' => $housingLevy,
            'helb_deduction' => $helbDeduction,
            'personal_relief' => $personalRelief,
            'insurance_relief' => $insuranceRelief,
            'pension_relief' => $pensionRelief,
            'loan_deductions' => $loanDeductions,
            'advance_deductions' => $advanceDeductions,
            'disciplinary_deductions' => $disciplinaryDeductions,
            'other_deductions' => $otherDeductions,
            'total_deductions' => $totalDeductions,
            'net_salary' => $netSalary,
        ];
    }

    /**
     * Calculate withholding tax for contractors
     */
    public static function calculateWithholdingTax(float $grossPayment, float $rate = 0.05): float
    {
        return round($grossPayment * $rate, 2);
    }

    /**
     * Calculate payroll for different employment types
     */
    public static function calculatePayrollByEmploymentType(array $salaryComponents, string $employmentType): array
    {
        $calculations = self::calculatePayroll($salaryComponents);
        
        // Apply employment type specific adjustments
        switch ($employmentType) {
            case 'contract':
            case 'freelance':
                // Apply withholding tax
                $withholdingTaxRate = $salaryComponents['withholding_tax_rate'] ?? 0.05;
                $calculations['withholding_tax'] = self::calculateWithholdingTax($calculations['gross_salary'], $withholdingTaxRate);
                $calculations['withholding_tax_rate'] = $withholdingTaxRate;
                
                // Check if statutory deductions apply
                if (!($salaryComponents['statutory_deductions_applicable'] ?? true)) {
                    $calculations['shif_deduction'] = 0;
                    $calculations['nhif_deduction'] = 0;
                    $calculations['nssf_employee'] = 0;
                    $calculations['housing_levy'] = 0;
                }
                break;
                
            case 'casual':
                // Casual workers typically don't get full statutory benefits
                $calculations['shif_deduction'] = 0;
                $calculations['nhif_deduction'] = 0;
                $calculations['nssf_employee'] = 0;
                $calculations['housing_levy'] = 0;
                break;
                
            case 'intern':
                // Interns typically have reduced statutory obligations
                $calculations['shif_deduction'] = 0;
                $calculations['nhif_deduction'] = 0;
                $calculations['nssf_employee'] = 0;
                break;
        }
        
        // Recalculate totals with employment type adjustments
        $calculations['total_deductions'] = $calculations['paye_tax'] + 
                                          $calculations['shif_deduction'] + 
                                          $calculations['nssf_employee'] + 
                                          $calculations['housing_levy'] + 
                                          ($calculations['withholding_tax'] ?? 0) +
                                          $calculations['helb_deduction'] + 
                                          $calculations['loan_deductions'] + 
                                          $calculations['advance_deductions'] + 
                                          $calculations['disciplinary_deductions'] + 
                                          $calculations['other_deductions'];
        
        $calculations['net_salary'] = $calculations['gross_salary'] - $calculations['total_deductions'];
        
        return $calculations;
    }

    /**
     * Calculate hourly-based payroll
     */
    public static function calculateHourlyPayroll(
        float $hourlyRate,
        float $regularHours,
        float $overtimeHours = 0,
        float $overtimeMultiplier = 1.5,
        array $additionalComponents = []
    ): array {
        $regularPay = $regularHours * $hourlyRate;
        $overtimePay = $overtimeHours * ($hourlyRate * $overtimeMultiplier);
        $basicSalary = $regularPay + $overtimePay;
        
        $components = array_merge([
            'basic_salary' => $basicSalary,
            'overtime_pay' => $overtimePay,
        ], $additionalComponents);
        
        return self::calculatePayroll($components);
    }

    /**
     * Calculate daily-based payroll
     */
    public static function calculateDailyPayroll(
        float $dailyRate,
        int $daysWorked,
        array $additionalComponents = []
    ): array {
        $basicSalary = $dailyRate * $daysWorked;
        
        $components = array_merge([
            'basic_salary' => $basicSalary,
        ], $additionalComponents);
        
        return self::calculatePayroll($components);
    }

    /**
     * Get employment type specific tax rules
     */
    public static function getEmploymentTypeTaxRules(string $employmentType): array
    {
        return match($employmentType) {
            'permanent' => [
                'paye_applicable' => true,
                'shif_applicable' => true, // Updated from nhif
                'nhif_applicable' => true, // Kept for backward compatibility
                'nssf_applicable' => true,
                'housing_levy_applicable' => true,
                'withholding_tax_applicable' => false,
                'withholding_tax_rate' => 0,
            ],
            'contract', 'freelance' => [
                'paye_applicable' => true,
                'shif_applicable' => false,
                'nhif_applicable' => false,
                'nssf_applicable' => false,
                'housing_levy_applicable' => false,
                'withholding_tax_applicable' => true,
                'withholding_tax_rate' => 0.05, // 5% standard rate
            ],
            'casual' => [
                'paye_applicable' => true,
                'shif_applicable' => false,
                'nhif_applicable' => false,
                'nssf_applicable' => false,
                'housing_levy_applicable' => false,
                'withholding_tax_applicable' => false,
                'withholding_tax_rate' => 0,
            ],
            'intern' => [
                'paye_applicable' => false, // Usually below tax threshold
                'shif_applicable' => false,
                'nhif_applicable' => false,
                'nssf_applicable' => false,
                'housing_levy_applicable' => false,
                'withholding_tax_applicable' => false,
                'withholding_tax_rate' => 0,
            ],
            'part_time' => [
                'paye_applicable' => true,
                'shif_applicable' => true,
                'nhif_applicable' => true,
                'nssf_applicable' => true,
                'housing_levy_applicable' => true,
                'withholding_tax_applicable' => false,
                'withholding_tax_rate' => 0,
            ],
            default => [
                'paye_applicable' => true,
                'shif_applicable' => true,
                'nhif_applicable' => true,
                'nssf_applicable' => true,
                'housing_levy_applicable' => true,
                'withholding_tax_applicable' => false,
                'withholding_tax_rate' => 0,
            ]
        };
    }
}