<?php

namespace App\Services;

use App\Models\Payroll;
use App\Models\PayrollApproval;
use App\Models\PayrollApprovalWorkflow;
use App\Models\User;

class PayrollApprovalService
{
    /**
     * Initialize approval workflow for a payroll
     */
    public function initializeApprovalWorkflow(Payroll $payroll): void
    {
        $workflow = PayrollApprovalWorkflow::getForOrganization($payroll->organization_id);
        
        // If no workflow configured, use default
        if ($workflow->isEmpty()) {
            $workflow = collect(PayrollApprovalWorkflow::getDefaultWorkflow());
        }

        foreach ($workflow as $step) {
            // Check if this level is required based on amount threshold
            $amountThreshold = is_object($step) ? $step->amount_threshold : ($step['amount_threshold'] ?? 0);
            $required = is_object($step) ? $step->required : ($step['required'] ?? true);
            
            if ($required && $payroll->gross_salary >= $amountThreshold) {
                PayrollApproval::create([
                    'payroll_id' => $payroll->id,
                    'level' => is_object($step) ? $step->level : $step['level'],
                    'approver_role' => is_object($step) ? $step->approver_role : $step['approver_role'],
                    'status' => 'pending',
                ]);
            }
        }
    }

    /**
     * Get current pending approval for a payroll
     */
    public function getCurrentPendingApproval(Payroll $payroll): ?PayrollApproval
    {
        return $payroll->approvals()
            ->where('status', 'pending')
            ->orderBy('level')
            ->orderBy('id')
            ->first();
    }

    /**
     * Check if user can approve current level
     */
    public function canUserApprove(Payroll $payroll, User $user): bool
    {
        $currentApproval = $this->getCurrentPendingApproval($payroll);
        
        if (!$currentApproval) {
            return false;
        }

        // Check if user has the required role
        return $user->hasRole($currentApproval->approver_role) || 
               $user->hasRole('Super Admin') || 
               $user->hasRole('Organization Admin');
    }

    /**
     * Approve current level
     */
    public function approveCurrentLevel(Payroll $payroll, User $user, ?string $comments = null): bool
    {
        $currentApproval = $this->getCurrentPendingApproval($payroll);
        
        if (!$currentApproval) {
            throw new \Exception('No pending approval found');
        }

        if (!$this->canUserApprove($payroll, $user)) {
            throw new \Exception('You do not have permission to approve this payroll');
        }

        $currentApproval->update([
            'approver_id' => $user->id,
            'status' => 'approved',
            'approved_at' => now(),
            'comments' => $comments,
        ]);

        // Check if all approvals are complete
        if ($this->isFullyApproved($payroll)) {
            $payroll->update([
                'approval_status' => 'approved',
                'approved_by' => $user->id,
                'approved_at' => now(),
            ]);
        }

        return true;
    }

    /**
     * Reject current level
     */
    public function rejectCurrentLevel(Payroll $payroll, User $user, string $comments): bool
    {
        $currentApproval = $this->getCurrentPendingApproval($payroll);
        
        if (!$currentApproval) {
            throw new \Exception('No pending approval found');
        }

        if (!$this->canUserApprove($payroll, $user)) {
            throw new \Exception('You do not have permission to reject this payroll');
        }

        $currentApproval->update([
            'approver_id' => $user->id,
            'status' => 'rejected',
            'approved_at' => now(),
            'comments' => $comments,
        ]);

        $payroll->update([
            'approval_status' => 'rejected',
        ]);

        return true;
    }

    /**
     * Check if payroll is fully approved
     */
    public function isFullyApproved(Payroll $payroll): bool
    {
        $totalApprovals = $payroll->approvals()->count();
        $approvedCount = $payroll->approvals()->where('status', 'approved')->count();
        
        return $totalApprovals > 0 && $totalApprovals === $approvedCount;
    }

    /**
     * Get approval progress
     */
    public function getApprovalProgress(Payroll $payroll): array
    {
        $approvals = $payroll->approvals()->orderBy('level')->get();
        
        return [
            'total_levels' => $approvals->count(),
            'approved_levels' => $approvals->where('status', 'approved')->count(),
            'current_level' => $this->getCurrentPendingApproval($payroll)?->level,
            'approvals' => $approvals,
        ];
    }
}
