<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class SecurityService
{
    /**
     * Check if IP is rate limited for login attempts
     */
    public function isLoginRateLimited(string $ip): bool
    {
        $key = 'login-attempts-' . $ip;
        return Cache::get($key, 0) >= 5;
    }

    /**
     * Increment login attempts for IP
     */
    public function incrementLoginAttempts(string $ip): int
    {
        $key = 'login-attempts-' . $ip;
        $attempts = Cache::get($key, 0) + 1;
        Cache::put($key, $attempts, now()->addMinutes(15));
        return $attempts;
    }

    /**
     * Clear login attempts for IP
     */
    public function clearLoginAttempts(string $ip): void
    {
        $key = 'login-attempts-' . $ip;
        Cache::forget($key);
    }

    /**
     * Lock user account
     */
    public function lockUserAccount(User $user, int $minutes = 60): void
    {
        $user->update([
            'locked_until' => now()->addMinutes($minutes),
            'failed_login_attempts' => $user->failed_login_attempts + 1
        ]);

        // Log security event
        activity()
            ->performedOn($user)
            ->withProperties([
                'action' => 'account_locked',
                'locked_until' => $user->locked_until,
                'reason' => 'excessive_failed_attempts'
            ])
            ->log('User account locked due to excessive failed login attempts');
    }

    /**
     * Check if user account is locked
     */
    public function isAccountLocked(User $user): bool
    {
        return $user->locked_until && now()->lessThan($user->locked_until);
    }

    /**
     * Unlock user account
     */
    public function unlockUserAccount(User $user): void
    {
        $user->update([
            'locked_until' => null,
            'failed_login_attempts' => 0
        ]);

        activity()
            ->performedOn($user)
            ->withProperties(['action' => 'account_unlocked'])
            ->log('User account unlocked');
    }

    /**
     * Generate secure password
     */
    public function generateSecurePassword(int $length = 12): string
    {
        return Str::password($length, true, true, true, false);
    }

    /**
     * Validate password strength
     */
    public function validatePasswordStrength(string $password): array
    {
        $errors = [];
        
        if (strlen($password) < 8) {
            $errors[] = 'Password must be at least 8 characters long';
        }
        
        if (!preg_match('/[A-Z]/', $password)) {
            $errors[] = 'Password must contain at least one uppercase letter';
        }
        
        if (!preg_match('/[a-z]/', $password)) {
            $errors[] = 'Password must contain at least one lowercase letter';
        }
        
        if (!preg_match('/[0-9]/', $password)) {
            $errors[] = 'Password must contain at least one number';
        }
        
        if (!preg_match('/[^A-Za-z0-9]/', $password)) {
            $errors[] = 'Password must contain at least one special character';
        }

        return $errors;
    }

    /**
     * Sanitize input data
     */
    public function sanitizeInput(string $input): string
    {
        return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
    }

    /**
     * Check for suspicious activity patterns
     */
    public function detectSuspiciousActivity(User $user, string $ip, string $userAgent): bool
    {
        // Check for rapid login attempts from different IPs
        $recentLogins = Cache::get("user-logins-{$user->id}", []);
        
        // Add current attempt
        $recentLogins[] = [
            'ip' => $ip,
            'user_agent' => $userAgent,
            'timestamp' => now()->timestamp
        ];
        
        // Keep only last 10 attempts
        $recentLogins = array_slice($recentLogins, -10);
        Cache::put("user-logins-{$user->id}", $recentLogins, now()->addHours(24));
        
        // Check for multiple IPs in short time
        $uniqueIps = collect($recentLogins)
            ->where('timestamp', '>', now()->subHours(1)->timestamp)
            ->pluck('ip')
            ->unique()
            ->count();
            
        return $uniqueIps > 3; // Suspicious if more than 3 different IPs in 1 hour
    }
}