<?php

namespace App\Services;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class TenantContext
{
    protected ?int $organizationId = null;
    protected bool $resolved = false;

    /**
     * Resolve tenant context
     */
    public function resolve(): void
    {
        if ($this->resolved) {
            return;
        }

        $user = Auth::user();
        
        if (!$user) {
            $this->resolved = true;
            return;
        }

        // System admins can switch context
        if ($user->isSystemAdmin()) {
            $this->organizationId = $this->getSystemAdminContext();
        } else {
            // Regular users are bound to their organization
            $this->organizationId = $user->organization_id;
        }

        // Bind to service container
        if ($this->organizationId) {
            app()->instance('tenant.organization_id', $this->organizationId);
        }

        $this->resolved = true;
    }

    /**
     * Get current organization ID
     */
    public function getOrganizationId(): ?int
    {
        if (!$this->resolved) {
            $this->resolve();
        }

        return $this->organizationId;
    }

    /**
     * Set organization context (for system admins)
     */
    public function setOrganizationId(?int $organizationId): void
    {
        $user = Auth::user();
        
        if (!$user || !$user->isSystemAdmin()) {
            throw new \Exception('Only system admins can switch organization context');
        }

        $this->organizationId = $organizationId;
        
        if ($organizationId) {
            app()->instance('tenant.organization_id', $organizationId);
            session(['current_organization_id' => $organizationId]);
        } else {
            session()->forget('current_organization_id');
        }
    }

    /**
     * Get system admin context from session or default
     */
    protected function getSystemAdminContext(): ?int
    {
        // Check session for current organization
        if (session()->has('current_organization_id')) {
            return (int) session('current_organization_id');
        }

        // Default to null for system admins (see all organizations)
        return null;
    }

    /**
     * Check if current user can access organization
     */
    public function canAccessOrganization(int $organizationId): bool
    {
        $user = Auth::user();
        
        if (!$user) {
            return false;
        }

        // System admins can access any organization
        if ($user->isSystemAdmin()) {
            return true;
        }

        // Regular users can only access their organization
        return $user->organization_id === $organizationId;
    }

    /**
     * Get organization data with caching
     */
    public function getOrganization(): ?\App\Models\Organization
    {
        if (!$this->organizationId) {
            return null;
        }

        return Cache::remember(
            "organization.{$this->organizationId}",
            now()->addHours(1),
            fn() => \App\Models\Organization::find($this->organizationId)
        );
    }
}