<?php

namespace App\Services;

use App\Models\ThemeSetting;
use Illuminate\Support\Facades\Cache;

class ThemeService
{
    /**
     * Request-level cache for theme name
     */
    protected ?string $themeName = null;

    /**
     * Request-level cache for theme config
     */
    protected ?array $themeConfig = null;

    /**
     * Get current theme name (cached per request)
     */
    public function getCurrentTheme(): string
    {
        // Return cached value if already resolved
        if ($this->themeName !== null) {
            return $this->themeName;
        }

        // Resolve theme name once per request
        $this->themeName = $this->resolveThemeName();
        
        return $this->themeName;
    }

    /**
     * Get theme configuration (cached per request)
     */
    public function getThemeConfig(): array
    {
        // Return cached value if already loaded
        if ($this->themeConfig !== null) {
            return $this->themeConfig;
        }

        // Load theme config once per request
        $themeName = $this->getCurrentTheme();
        $this->themeConfig = $this->loadThemeConfig($themeName);
        
        return $this->themeConfig;
    }

    /**
     * Resolve theme name based on user context
     */
    protected function resolveThemeName(): string
    {
        // Skip for unauthenticated users
        if (!auth()->check()) {
            return config('theme.default', 'default');
        }

        $user = auth()->user();

        // Check user personal preference
        if ($user->theme_preference) {
            return $user->theme_preference;
        }

        // Check if system admin
        if ($user->is_system_admin ?? false) {
            return 'system_admin';
        }

        // Check organization theme (from database)
        if ($user->organization_id) {
            $themeSetting = ThemeSetting::getForOrganization($user->organization_id);
            if ($themeSetting) {
                return $themeSetting->base_theme;
            }
        }

        return config('theme.default', 'default');
    }

    /**
     * Load theme configuration with database overrides
     */
    protected function loadThemeConfig(string $themeName): array
    {
        // Start with base theme from config
        $theme = config("theme.themes.{$themeName}");

        // Fallback to default theme if not found
        if (!$theme) {
            $theme = config('theme.themes.default');
        }

        // Ensure we have a valid theme array
        if (!is_array($theme)) {
            $theme = $this->getDefaultThemeStructure();
        }

        // Skip database lookup for unauthenticated users
        if (!auth()->check() || !auth()->user()->organization_id) {
            return $theme;
        }

        // Check for organization customizations
        $themeSetting = ThemeSetting::getForOrganization(auth()->user()->organization_id);

        if ($themeSetting) {
            // Merge database customizations with base theme
            $theme = $themeSetting->getThemeConfig();
        }

        return $theme;
    }

    /**
     * Get default theme structure as fallback
     */
    protected function getDefaultThemeStructure(): array
    {
        return [
            'name' => 'Default Theme',
            'colors' => [
                'primary' => '#030213',
                'secondary' => '#FF6B35',
                'accent' => '#00a09d',
                'success' => '#28a745',
                'warning' => '#ffc107',
                'danger' => '#dc3545',
                'info' => '#17a2b8',
                'light' => '#f8f9fa',
                'dark' => '#343a40',
            ],
            'sidebar' => [
                'background' => '#030213',
                'text' => 'rgba(255, 255, 255, 0.8)',
                'hover' => '#FF6B35',
                'active' => '#FF6B35',
            ],
        ];
    }

    /**
     * Generate static CSS file for organization theme
     */
    public function generateStaticCSS(int $organizationId): string
    {
        $themeSetting = ThemeSetting::getForOrganization($organizationId);
        
        if (!$themeSetting) {
            return '';
        }

        $theme = $themeSetting->getThemeConfig();
        
        $css = ":root {\n";
        
        // Add color variables
        if (isset($theme['colors'])) {
            foreach ($theme['colors'] as $key => $value) {
                $css .= "    --color-{$key}: {$value};\n";
            }
        }
        
        // Add sidebar variables
        if (isset($theme['sidebar'])) {
            $css .= "    --sidebar-bg: {$theme['sidebar']['background']};\n";
            $css .= "    --sidebar-text: {$theme['sidebar']['text']};\n";
            $css .= "    --sidebar-hover: {$theme['sidebar']['hover']};\n";
            $css .= "    --sidebar-active: {$theme['sidebar']['active']};\n";
        }
        
        // Add font variables
        if (isset($theme['fonts']['primary']) && $theme['fonts']['primary']) {
            $css .= "    --font-primary: {$theme['fonts']['primary']};\n";
        }
        if (isset($theme['fonts']['heading']) && $theme['fonts']['heading']) {
            $css .= "    --font-heading: {$theme['fonts']['heading']};\n";
        }
        
        $css .= "}\n\n";
        
        // Add font styles
        if (isset($theme['fonts']['primary']) && $theme['fonts']['primary']) {
            $css .= "body {\n";
            $css .= "    font-family: {$theme['fonts']['primary']}, sans-serif;\n";
            $css .= "}\n\n";
        }
        
        if (isset($theme['fonts']['heading']) && $theme['fonts']['heading']) {
            $css .= "h1, h2, h3, h4, h5, h6 {\n";
            $css .= "    font-family: {$theme['fonts']['heading']}, sans-serif;\n";
            $css .= "}\n\n";
        }
        
        // Add custom CSS
        if (isset($theme['custom_css']) && $theme['custom_css']) {
            $css .= "/* Custom CSS */\n";
            $css .= $theme['custom_css'] . "\n";
        }
        
        return $css;
    }

    /**
     * Clear theme cache for organization
     */
    public function clearCache(int $organizationId): void
    {
        Cache::forget("theme_settings.org.{$organizationId}");
    }
}
