<?php

namespace App\Traits;

use App\Services\AuditService;
use Illuminate\Database\Eloquent\Model;

trait Auditable
{
    protected static function bootAuditable(): void
    {
        // Track model creation
        static::created(function (Model $model) {
            if (self::shouldAudit($model, 'created')) {
                AuditService::logModelEvent(
                    'created',
                    $model,
                    [],
                    $model->getAuditableAttributes()
                );
            }
        });

        // Track model updates
        static::updated(function (Model $model) {
            if (self::shouldAudit($model, 'updated')) {
                $oldValues = [];
                $newValues = [];
                
                foreach ($model->getDirty() as $key => $newValue) {
                    if (in_array($key, $model->getAuditableAttributes())) {
                        $oldValues[$key] = $model->getOriginal($key);
                        $newValues[$key] = $newValue;
                    }
                }

                if (!empty($oldValues) || !empty($newValues)) {
                    AuditService::logModelEvent('updated', $model, $oldValues, $newValues);
                }
            }
        });

        // Track model deletion
        static::deleted(function (Model $model) {
            if (self::shouldAudit($model, 'deleted')) {
                AuditService::logModelEvent(
                    'deleted',
                    $model,
                    $model->getAuditableAttributes(),
                    []
                );
            }
        });

        // Track model restoration (if using soft deletes)
        if (method_exists(static::class, 'restored')) {
            static::restored(function (Model $model) {
                if (self::shouldAudit($model, 'restored')) {
                    AuditService::logModelEvent('restored', $model);
                }
            });
        }
    }

    /**
     * Get attributes that should be audited
     */
    public function getAuditableAttributes(): array
    {
        if (property_exists($this, 'auditableAttributes')) {
            return $this->auditableAttributes;
        }

        // Default: audit all fillable attributes except sensitive ones
        $excluded = $this->getAuditExcluded();
        return array_diff($this->getFillable(), $excluded);
    }

    /**
     * Get attributes that should be excluded from auditing
     */
    public function getAuditExcluded(): array
    {
        return property_exists($this, 'auditExcluded') 
            ? $this->auditExcluded 
            : ['password', 'remember_token', 'email_verified_at'];
    }

    /**
     * Determine if the model should be audited for the given event
     */
    protected static function shouldAudit(Model $model, string $event): bool
    {
        // Skip if auditing is disabled for this model
        if (property_exists($model, 'auditingEnabled') && !$model->auditingEnabled) {
            return false;
        }

        // Skip if specific events are disabled
        if (property_exists($model, 'auditEvents')) {
            return in_array($event, $model->auditEvents);
        }

        // Skip during seeding or testing if configured
        if (app()->runningInConsole() && !config('audit.console_enabled', false)) {
            return false;
        }

        return true;
    }

    /**
     * Manually log an audit event for this model
     */
    public function auditLog(string $eventType, string $description, array $metadata = []): void
    {
        AuditService::log(
            eventType: $eventType,
            description: $description,
            auditable: $this,
            metadata: $metadata
        );
    }
}