<?php

namespace App\Traits;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;

trait HasTenantScope
{
    /**
     * Boot the trait
     */
    protected static function bootHasTenantScope()
    {
        // Add global scope to filter by organization
        static::addGlobalScope('organization', function (Builder $builder) {
            $organizationId = static::getCurrentOrganizationId();
            
            if ($organizationId) {
                $builder->where($builder->getModel()->getTable() . '.organization_id', $organizationId);
            }
        });

        // Automatically set organization_id when creating records
        static::creating(function ($model) {
            $organizationId = static::getCurrentOrganizationId();
            
            if ($organizationId && !$model->organization_id) {
                $model->organization_id = $organizationId;
            }
        });
    }

    /**
     * Get current organization ID from various sources
     */
    protected static function getCurrentOrganizationId(): ?int
    {
        // Use tenant context service if available
        if (app()->bound('tenant.context')) {
            return app('tenant.context')->getOrganizationId();
        }

        // Fallback to service container
        if (app()->bound('tenant.organization_id')) {
            return app('tenant.organization_id');
        }

        // Final fallback to authenticated user's organization
        if (Auth::check() && Auth::user()->organization_id) {
            $user = Auth::user();
            // Don't auto-scope for system admins unless explicitly set
            $isSystemAdmin = isset($user->is_system_admin) ? $user->is_system_admin : false;
            if (!$isSystemAdmin) {
                return $user->organization_id;
            }
        }

        return null;
    }

    /**
     * Get the organization that owns this model
     */
    public function organization()
    {
        return $this->belongsTo(\App\Models\Organization::class);
    }

    /**
     * Scope query to specific organization
     */
    public function scopeForOrganization(Builder $query, $organizationId)
    {
        return $query->where('organization_id', $organizationId);
    }

    /**
     * Remove organization scope temporarily
     */
    public function scopeWithoutOrganizationScope(Builder $query)
    {
        return $query->withoutGlobalScope('organization');
    }

    /**
     * Scope to current user's organization (explicit)
     */
    public function scopeForCurrentOrganization(Builder $query)
    {
        $organizationId = static::getCurrentOrganizationId();
        
        if ($organizationId) {
            return $query->where('organization_id', $organizationId);
        }
        
        return $query;
    }

    /**
     * Check if model belongs to current organization
     */
    public function belongsToCurrentOrganization(): bool
    {
        $currentOrgId = static::getCurrentOrganizationId();
        return $currentOrgId && $this->organization_id == $currentOrgId;
    }
}