<?php

namespace App\Traits;

use App\Models\Organization;
use App\Services\OrganizationMailService;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

trait SendsOrganizationMail
{
    /**
     * Send email using organization's email configuration
     * 
     * @param Organization|null $organization
     * @param string|array $to
     * @param \Illuminate\Mail\Mailable $mailable
     * @return bool
     */
    protected function sendOrganizationMail($organization, $to, $mailable): bool
    {
        try {
            // Configure organization email settings if custom mail is enabled
            if ($organization && $organization->use_custom_mail) {
                OrganizationMailService::configureForOrganization($organization);
            }
            
            Mail::to($to)->send($mailable);
            
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send organization email', [
                'organization_id' => $organization?->id,
                'to' => $to,
                'mailable' => get_class($mailable),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return false;
        }
    }

    /**
     * Send email using system email configuration (for system admin actions)
     * 
     * @param string|array $to
     * @param \Illuminate\Mail\Mailable $mailable
     * @return bool
     */
    protected function sendSystemMail($to, $mailable): bool
    {
        try {
            Log::info('Attempting to send system email', [
                'to' => $to,
                'mailable' => get_class($mailable),
                'mail_driver' => config('mail.default'),
                'mail_host' => config('mail.mailers.smtp.host'),
            ]);
            
            // Use default .env email configuration
            Mail::to($to)->send($mailable);
            
            Log::info('System email sent successfully', [
                'to' => $to,
                'mailable' => get_class($mailable),
            ]);
            
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send system email', [
                'to' => $to,
                'mailable' => get_class($mailable),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return false;
        }
    }

    /**
     * Get current user's organization
     * 
     * @return Organization|null
     */
    protected function getCurrentOrganization(): ?Organization
    {
        $user = auth()->user();
        return $user ? $user->organization : null;
    }
}
