<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Create candidates table
        Schema::create('candidates', function (Blueprint $table) {
            $table->id();
            $table->string('email')->unique();
            $table->timestamp('email_verified_at')->nullable();
            $table->string('password');
            $table->string('first_name', 100);
            $table->string('last_name', 100);
            $table->string('phone', 20)->nullable();
            $table->rememberToken();
            $table->timestamps();

            // Indexes for performance
            $table->index('email');
        });

        // Create job_postings table
        Schema::create('job_postings', function (Blueprint $table) {
            $table->id();
            $table->string('title');
            $table->foreignId('department_id')->constrained()->onDelete('cascade');
            $table->foreignId('position_id')->nullable()->constrained()->onDelete('set null');
            $table->string('location');
            $table->enum('employment_type', ['full-time', 'part-time', 'contract', 'internship'])->default('full-time');
            $table->text('description');
            $table->text('requirements');
            $table->text('responsibilities');
            $table->string('salary_range', 100)->nullable();
            $table->enum('status', ['draft', 'active', 'closed'])->default('draft');
            $table->string('public_token', 64)->unique();
            $table->timestamp('closes_at')->nullable();
            $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            // Indexes for performance
            $table->index(['organization_id', 'status']);
            $table->index(['status', 'closes_at']);
            $table->index('public_token');
            $table->index('created_by');
        });

        // Create applications table
        Schema::create('applications', function (Blueprint $table) {
            $table->id();
            $table->foreignId('job_posting_id')->constrained()->onDelete('cascade');
            $table->unsignedBigInteger('candidate_id')->nullable(); // Remove foreign key constraint for now
            $table->string('first_name', 100);
            $table->string('last_name', 100);
            $table->string('email');
            $table->string('phone', 20);
            $table->string('cv_path');
            $table->text('cover_letter')->nullable();
            $table->enum('status', ['new', 'screening', 'shortlisted', 'interviewed', 'rejected', 'offered'])->default('new');
            $table->decimal('screening_score', 5, 2)->nullable();
            $table->text('notes')->nullable();
            $table->integer('updated_count')->default(0);
            $table->timestamp('last_updated_at')->nullable();
            $table->string('tracking_token', 64)->unique();
            $table->timestamp('applied_at');
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            // Indexes for performance
            $table->index(['job_posting_id', 'status']);
            $table->index(['organization_id', 'status']);
            $table->index(['email', 'job_posting_id']); // Prevent duplicates
            $table->index('tracking_token');
            $table->index('applied_at');
        });

        // Create interviews table
        Schema::create('interviews', function (Blueprint $table) {
            $table->id();
            $table->foreignId('application_id')->constrained()->onDelete('cascade');
            $table->timestamp('scheduled_at');
            $table->string('location')->nullable();
            $table->string('meeting_link', 500)->nullable();
            $table->foreignId('interviewer_id')->constrained('users')->onDelete('cascade');
            $table->enum('status', ['scheduled', 'completed', 'cancelled'])->default('scheduled');
            $table->decimal('total_score', 5, 2)->nullable();
            $table->text('notes')->nullable();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            // Indexes for performance
            $table->index(['application_id', 'status']);
            $table->index(['interviewer_id', 'scheduled_at']);
            $table->index(['organization_id', 'status']);
            $table->index('scheduled_at');
        });

        // Create interview_questions table
        Schema::create('interview_questions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('interview_id')->constrained()->onDelete('cascade');
            $table->text('question');
            $table->decimal('score', 5, 2)->nullable();
            $table->decimal('max_score', 5, 2)->default(10);
            $table->text('notes')->nullable();
            $table->timestamps();

            // Indexes for performance
            $table->index('interview_id');
        });

        // Create application_status_histories table
        Schema::create('application_status_histories', function (Blueprint $table) {
            $table->id();
            $table->foreignId('application_id')->constrained()->onDelete('cascade');
            $table->string('from_status', 50)->nullable();
            $table->string('to_status', 50);
            $table->foreignId('changed_by')->constrained('users')->onDelete('cascade');
            $table->text('notes')->nullable();
            $table->timestamps();

            // Indexes for performance
            $table->index(['application_id', 'created_at']);
            $table->index('changed_by');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('application_status_histories');
        Schema::dropIfExists('interview_questions');
        Schema::dropIfExists('interviews');
        Schema::dropIfExists('applications');
        Schema::dropIfExists('job_postings');
        Schema::dropIfExists('candidates');
    }
};