<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Update existing performance_reviews table
        Schema::table('performance_reviews', function (Blueprint $table) {
            // Remove unnecessary individual rating fields - we'll use a more flexible approach
            $table->dropColumn([
                'technical_skills_rating',
                'communication_rating', 
                'teamwork_rating',
                'productivity_rating',
                'punctuality_rating'
            ]);
            
            // Add new fields for enhanced functionality
            $table->enum('review_type', ['annual', 'mid_year', 'quarterly', 'probation', 'project_based'])->default('annual')->after('review_period');
            $table->boolean('is_self_assessment')->default(false)->after('review_type');
            $table->json('performance_ratings')->nullable()->after('overall_rating'); // Store flexible ratings
            $table->decimal('goal_achievement_score', 5, 2)->nullable()->after('performance_ratings');
            $table->timestamp('employee_acknowledged_at')->nullable()->after('acknowledged_at');
            $table->timestamp('due_date')->nullable()->after('review_date');
            $table->text('manager_comments')->nullable()->after('comments');
            $table->text('hr_comments')->nullable()->after('manager_comments');
            $table->json('development_areas')->nullable()->after('areas_of_improvement'); // Structured development areas
        });

        // Create performance_goals table
        Schema::create('performance_goals', function (Blueprint $table) {
            $table->id();
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->foreignId('performance_review_id')->nullable()->constrained()->onDelete('cascade');
            $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
            $table->string('title');
            $table->text('description');
            $table->enum('type', ['individual', 'team', 'company', 'development', 'project']);
            $table->enum('priority', ['low', 'medium', 'high', 'critical'])->default('medium');
            $table->date('start_date');
            $table->date('target_date');
            $table->date('completion_date')->nullable();
            $table->enum('status', ['draft', 'active', 'completed', 'cancelled', 'overdue'])->default('draft');
            $table->integer('weight_percentage')->default(0); // For weighted goal scoring
            $table->decimal('target_value', 10, 2)->nullable(); // For quantifiable goals
            $table->decimal('achieved_value', 10, 2)->nullable();
            $table->string('measurement_unit')->nullable(); // e.g., 'sales', 'projects', 'percentage'
            $table->text('success_criteria');
            $table->json('milestones')->nullable(); // Store milestone data
            $table->decimal('progress_percentage', 5, 2)->default(0);
            $table->text('notes')->nullable();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->index(['employee_id', 'status']);
            $table->index(['performance_review_id']);
            $table->index(['target_date', 'status']);
        });

        // Create performance_goal_progress table for tracking updates
        Schema::create('performance_goal_progress', function (Blueprint $table) {
            $table->id();
            $table->foreignId('performance_goal_id')->constrained()->onDelete('cascade');
            $table->foreignId('updated_by')->constrained('users')->onDelete('cascade');
            $table->decimal('progress_percentage', 5, 2);
            $table->decimal('achieved_value', 10, 2)->nullable();
            $table->text('update_notes');
            $table->json('milestone_updates')->nullable();
            $table->date('update_date');
            $table->timestamps();

            $table->index(['performance_goal_id', 'update_date']);
        });

        // Create performance_competencies table
        Schema::create('performance_competencies', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->text('description');
            $table->enum('category', ['technical', 'behavioral', 'leadership', 'core']);
            $table->json('skill_levels'); // Define skill levels (1-5 with descriptions)
            $table->boolean('is_active')->default(true);
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->index(['category', 'is_active']);
        });

        // Create performance_competency_ratings table
        Schema::create('performance_competency_ratings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('performance_review_id')->constrained()->onDelete('cascade');
            $table->foreignId('performance_competency_id')->constrained()->onDelete('cascade');
            $table->integer('self_rating')->nullable(); // 1-5 scale
            $table->integer('manager_rating')->nullable(); // 1-5 scale
            $table->text('self_comments')->nullable();
            $table->text('manager_comments')->nullable();
            $table->decimal('weight', 5, 2)->default(1.00); // Competency weight in overall score
            $table->timestamps();

            $table->unique(['performance_review_id', 'performance_competency_id'], 'unique_review_competency');
        });

        // Create performance_self_assessments table
        Schema::create('performance_self_assessments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->foreignId('performance_review_id')->nullable()->constrained()->onDelete('cascade');
            $table->date('assessment_date');
            $table->enum('period', ['Q1', 'Q2', 'Q3', 'Q4', 'Mid-Year', 'Annual']);
            $table->json('achievements'); // Key accomplishments
            $table->json('challenges_faced'); // Challenges and how they were overcome
            $table->text('strengths_identified');
            $table->text('areas_for_development');
            $table->json('learning_goals'); // What they want to learn/improve
            $table->json('career_aspirations'); // Career goals and interests
            $table->text('support_needed'); // What support they need from manager/organization
            $table->decimal('overall_satisfaction', 3, 2)->nullable(); // Job satisfaction rating
            $table->text('additional_feedback')->nullable();
            $table->enum('status', ['draft', 'submitted', 'reviewed'])->default('draft');
            $table->timestamp('submitted_at')->nullable();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->index(['employee_id', 'period']);
            $table->index(['status', 'submitted_at']);
        });

        // Create performance_development_plans table
        Schema::create('performance_development_plans', function (Blueprint $table) {
            $table->id();
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->foreignId('performance_review_id')->nullable()->constrained()->onDelete('cascade');
            $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
            $table->string('title');
            $table->text('description');
            $table->json('development_areas'); // Areas identified for development
            $table->json('action_items'); // Specific actions to take
            $table->json('resources_needed'); // Training, mentoring, etc.
            $table->date('start_date');
            $table->date('target_completion_date');
            $table->date('actual_completion_date')->nullable();
            $table->enum('status', ['draft', 'active', 'completed', 'cancelled'])->default('draft');
            $table->text('success_metrics');
            $table->text('progress_notes')->nullable();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->index(['employee_id', 'status']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('performance_development_plans');
        Schema::dropIfExists('performance_self_assessments');
        Schema::dropIfExists('performance_competency_ratings');
        Schema::dropIfExists('performance_competencies');
        Schema::dropIfExists('performance_goal_progress');
        Schema::dropIfExists('performance_goals');
        
        // Restore original performance_reviews structure
        Schema::table('performance_reviews', function (Blueprint $table) {
            $table->dropColumn([
                'review_type',
                'is_self_assessment',
                'performance_ratings',
                'goal_achievement_score',
                'employee_acknowledged_at',
                'due_date',
                'manager_comments',
                'hr_comments',
                'development_areas'
            ]);
            
            // Restore original rating columns
            $table->integer('technical_skills_rating')->nullable()->after('overall_rating');
            $table->integer('communication_rating')->nullable()->after('technical_skills_rating');
            $table->integer('teamwork_rating')->nullable()->after('communication_rating');
            $table->integer('productivity_rating')->nullable()->after('teamwork_rating');
            $table->integer('punctuality_rating')->nullable()->after('productivity_rating');
        });
    }
};