<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // ============================================
        // PAYROLL SETTINGS
        // ============================================
        
        Schema::create('payroll_settings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            
            // PAYE Tax Settings
            $table->json('paye_bands')->nullable();
            $table->decimal('personal_relief', 10, 2)->default(2400);
            $table->decimal('insurance_relief_rate', 5, 4)->default(0.15);
            $table->decimal('insurance_relief_max', 10, 2)->default(5000);
            $table->decimal('pension_relief_max', 10, 2)->default(20000);
            $table->decimal('pension_relief_rate', 5, 4)->default(0.30);
            
            // SHIF Settings (replaced NHIF)
            $table->decimal('shif_rate', 5, 4)->default(0.0275);
            $table->decimal('shif_min_contribution', 10, 2)->default(300);
            $table->boolean('shif_tax_deductible')->default(true);
            
            // NSSF Settings (2026 rates)
            $table->decimal('nssf_employee_rate', 5, 4)->default(0.06);
            $table->decimal('nssf_employer_rate', 5, 4)->default(0.06);
            $table->decimal('nssf_lower_limit', 10, 2)->default(9000);
            $table->decimal('nssf_upper_limit', 10, 2)->default(108000);
            $table->decimal('nssf_max_employee_contribution', 10, 2)->default(6480);
            $table->decimal('nssf_max_employer_contribution', 10, 2)->default(6480);
            $table->decimal('nssf_min_contribution', 10, 2)->default(540);
            
            // Housing Levy Settings (2026 rates)
            $table->decimal('housing_levy_employee_rate', 5, 4)->default(0.015);
            $table->decimal('housing_levy_employer_rate', 5, 4)->default(0.015);
            $table->boolean('housing_levy_tax_deductible')->default(true);
            $table->boolean('housing_levy_no_max_cap')->default(true);
            
            // Default Allowances
            $table->decimal('default_house_allowance_rate', 5, 4)->default(0.15);
            $table->decimal('default_transport_allowance_rate', 5, 4)->default(0.05);
            $table->decimal('max_transport_allowance', 10, 2)->default(5000);
            $table->decimal('default_medical_allowance', 10, 2)->default(2000);
            
            // Working Days Configuration
            $table->integer('default_monthly_working_days')->default(22);
            $table->integer('default_weekly_working_days')->default(5);
            $table->integer('default_hours_per_day')->default(8);
            
            // Compliance Deadlines
            $table->integer('kra_submission_deadline')->default(9);
            $table->integer('shif_submission_deadline')->default(15);
            $table->integer('nssf_submission_deadline')->default(15);
            $table->integer('housing_levy_submission_deadline')->default(9);
            
            // Configuration Metadata
            $table->string('config_version')->default('2026.02');
            $table->timestamp('last_regulatory_update')->default(now());
            $table->json('regulatory_change_log')->nullable();
            
            $table->timestamps();
            
            $table->unique('organization_id');
        });

        // ============================================
        // PAYROLL APPROVALS
        // ============================================
        
        Schema::create('payroll_approvals', function (Blueprint $table) {
            $table->id();
            $table->foreignId('payroll_id')->constrained()->onDelete('cascade');
            $table->integer('level')->default(1);
            $table->string('approver_role');
            $table->foreignId('approver_id')->nullable()->constrained('users')->onDelete('set null');
            $table->enum('status', ['pending', 'approved', 'rejected'])->default('pending');
            $table->timestamp('approved_at')->nullable();
            $table->text('comments')->nullable();
            $table->timestamps();

            $table->index(['payroll_id', 'level']);
            $table->index(['approver_id', 'status']);
        });

        // ============================================
        // PAYROLL APPROVAL WORKFLOWS
        // ============================================
        
        Schema::create('payroll_approval_workflows', function (Blueprint $table) {
            $table->id();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->integer('level')->default(1);
            $table->string('approver_role');
            $table->decimal('amount_threshold', 15, 2)->nullable();
            $table->boolean('required')->default(true);
            $table->integer('order')->default(1);
            $table->timestamps();

            $table->index(['organization_id', 'level']);
            $table->unique(['organization_id', 'level', 'order']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('payroll_approval_workflows');
        Schema::dropIfExists('payroll_approvals');
        Schema::dropIfExists('payroll_settings');
    }
};
