<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // ============================================
        // LOAN SETTINGS
        // ============================================
        
        Schema::create('loan_settings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            
            // Loan Limits
            $table->decimal('max_loan_amount', 15, 2)->default(500000);
            $table->integer('max_loan_term_months')->default(36);
            $table->integer('min_employment_tenure_months')->default(6);
            $table->integer('max_outstanding_loans')->default(2);
            $table->decimal('loan_to_salary_ratio', 5, 2)->default(3.0);
            
            // Interest Rates
            $table->json('default_interest_rates')->nullable();
            
            // Approval Settings
            $table->decimal('auto_approve_below_amount', 15, 2)->default(10000);
            $table->decimal('require_guarantor_above_amount', 15, 2)->default(100000);
            $table->boolean('require_department_head_approval')->default(true);
            $table->boolean('require_finance_approval')->default(true);
            $table->boolean('require_ceo_approval_above_threshold')->default(true);
            $table->decimal('ceo_approval_threshold', 15, 2)->default(200000);
            
            // Eligibility
            $table->boolean('check_credit_score')->default(false);
            $table->boolean('allow_multiple_loans')->default(true);
            $table->boolean('allow_loan_with_pending_loan')->default(false);
            
            // Repayment
            $table->boolean('allow_early_repayment')->default(true);
            $table->boolean('allow_payment_holidays')->default(false);
            $table->integer('max_payment_holidays')->default(2);
            $table->decimal('early_repayment_penalty_rate', 5, 2)->default(0);
            
            // Notifications
            $table->boolean('notify_on_approval')->default(true);
            $table->boolean('notify_before_deduction')->default(true);
            $table->integer('notification_days_before')->default(3);
            
            // Metadata
            $table->text('loan_policy_document')->nullable();
            $table->text('terms_and_conditions')->nullable();
            
            $table->timestamps();
            
            $table->unique('organization_id');
        });

        // ============================================
        // EMPLOYEE LOANS
        // ============================================
        
        Schema::create('employee_loans', function (Blueprint $table) {
            $table->id();
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->string('loan_number')->unique();
            $table->enum('loan_type', ['salary_advance', 'emergency', 'personal', 'education', 'housing', 'other'])->default('personal');
            $table->decimal('principal_amount', 15, 2);
            $table->decimal('interest_rate', 5, 2)->default(0);
            $table->integer('repayment_period_months');
            $table->decimal('monthly_deduction', 15, 2);
            $table->decimal('total_amount', 15, 2);
            $table->decimal('amount_paid', 15, 2)->default(0);
            $table->decimal('balance', 15, 2);
            $table->date('disbursement_date');
            $table->date('first_deduction_date');
            $table->date('expected_completion_date');
            $table->date('actual_completion_date')->nullable();
            $table->enum('status', ['pending', 'approved', 'disbursed', 'active', 'completed', 'defaulted', 'cancelled'])->default('pending');
            $table->text('purpose')->nullable();
            $table->text('terms_conditions')->nullable();
            $table->foreignId('approved_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamp('approved_at')->nullable();
            $table->text('approval_notes')->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->index(['employee_id', 'status']);
            $table->index(['organization_id', 'status']);
        });

        // ============================================
        // LOAN REPAYMENTS
        // ============================================
        
        Schema::create('loan_repayments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('loan_id')->constrained('employee_loans')->onDelete('cascade');
            $table->foreignId('payroll_id')->nullable()->constrained()->onDelete('set null');
            $table->decimal('amount', 15, 2);
            $table->decimal('principal_portion', 15, 2);
            $table->decimal('interest_portion', 15, 2);
            $table->date('payment_date');
            $table->enum('payment_method', ['payroll_deduction', 'direct_payment', 'bank_transfer'])->default('payroll_deduction');
            $table->text('notes')->nullable();
            $table->timestamps();

            $table->index(['loan_id', 'payment_date']);
        });

        // ============================================
        // EMPLOYEE ADVANCES
        // ============================================
        
        Schema::create('employee_advances', function (Blueprint $table) {
            $table->id();
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->string('advance_number')->unique();
            $table->decimal('amount', 15, 2);
            $table->decimal('amount_repaid', 15, 2)->default(0);
            $table->decimal('balance', 15, 2);
            $table->integer('repayment_months')->default(1);
            $table->decimal('monthly_deduction', 15, 2);
            $table->date('advance_date');
            $table->date('first_deduction_date');
            $table->enum('status', ['pending', 'approved', 'active', 'completed', 'cancelled'])->default('pending');
            $table->text('reason')->nullable();
            $table->foreignId('approved_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamp('approved_at')->nullable();
            $table->timestamps();
            $table->softDeletes();

            $table->index(['employee_id', 'status']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('employee_advances');
        Schema::dropIfExists('loan_repayments');
        Schema::dropIfExists('employee_loans');
        Schema::dropIfExists('loan_settings');
    }
};
