<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Employee;
use App\Models\Attendance;
use Carbon\Carbon;

class AttendanceSeeder extends Seeder
{
    public function run(): void
    {
        $employees = Employee::all();
        
        // Create attendance for January 2026
        $year = 2026;
        $month = 1;
        $daysInMonth = Carbon::create($year, $month)->daysInMonth;
        
        foreach ($employees as $employee) {
            for ($day = 1; $day <= $daysInMonth; $day++) {
                $date = Carbon::create($year, $month, $day);
                
                // Skip weekends
                if ($date->isWeekend()) {
                    continue;
                }
                
                // 90% attendance rate (some random absences)
                if (rand(1, 100) > 90) {
                    continue;
                }
                
                $checkInHour = 8;
                $checkInMinute = rand(0, 30);
                $checkOutHour = 17;
                $checkOutMinute = rand(0, 30);
                
                // Some overtime (20% chance)
                if (rand(1, 100) <= 20) {
                    $checkOutHour = 19;
                    $checkOutMinute = rand(0, 59); // Fixed: was 60, should be 59
                }
                
                $checkIn = sprintf('%02d:%02d:00', $checkInHour, $checkInMinute);
                $checkOut = sprintf('%02d:%02d:00', $checkOutHour, $checkOutMinute);
                
                // Calculate worked hours manually to avoid Carbon issues
                $checkInTime = Carbon::createFromFormat('H:i:s', $checkIn);
                $checkOutTime = Carbon::createFromFormat('H:i:s', $checkOut);
                $workedMinutes = $checkInTime->diffInMinutes($checkOutTime);
                
                Attendance::create([
                    'employee_id' => $employee->id,
                    'date' => $date->toDateString(),
                    'check_in' => $checkIn,
                    'check_out' => $checkOut,
                    'worked_hours' => round($workedMinutes / 60, 2),
                    'regular_hours' => min(8, round($workedMinutes / 60, 2)),
                    'overtime_hours' => max(0, round(($workedMinutes - 480) / 60, 2)), // 480 minutes = 8 hours
                    'status' => 'present',
                    'approval_status' => 'approved',
                    'organization_id' => $employee->organization_id ?? 1,
                ]);
            }
        }
        
        $this->command->info('Attendance records created for January 2026!');
    }
}