<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Employee;
use App\Models\Contract;
use Carbon\Carbon;

class ContractSeeder extends Seeder
{
    public function run(): void
    {
        $employees = Employee::all();
        
        $employmentTypes = ['permanent', 'contract', 'freelance', 'casual', 'intern', 'part_time'];
        $paymentFrequencies = ['monthly', 'hourly', 'daily'];
        
        foreach ($employees as $employee) {
            $employmentType = $employmentTypes[array_rand($employmentTypes)];
            $paymentFrequency = $paymentFrequencies[array_rand($paymentFrequencies)];
            
            // Adjust payment frequency based on employment type
            if ($employmentType === 'permanent') {
                $paymentFrequency = 'monthly';
            } elseif ($employmentType === 'casual') {
                $paymentFrequency = 'daily';
            } elseif (in_array($employmentType, ['intern', 'part_time'])) {
                $paymentFrequency = 'hourly';
            }
            
            $salary = match($employmentType) {
                'permanent' => rand(50000, 150000),
                'contract' => $paymentFrequency === 'monthly' ? rand(40000, 120000) : rand(30000, 80000),
                'freelance' => rand(25000, 60000),
                'casual' => rand(15000, 40000),
                'intern' => rand(10000, 25000),
                'part_time' => rand(20000, 50000),
            };
            
            $hourlyRate = match($paymentFrequency) {
                'hourly' => rand(500, 2000),
                default => null,
            };
            
            $dailyRate = match($paymentFrequency) {
                'daily' => rand(2000, 8000),
                default => null,
            };
            
            Contract::create([
                'contract_reference' => 'CON-' . $employee->employee_code . '-' . date('Y'),
                'employee_id' => $employee->id,
                'employment_type' => $employmentType,
                'contract_type' => $employmentType === 'permanent' ? 'permanent' : 'fixed_term',
                'start_date' => $employee->joining_date,
                'end_date' => $employmentType === 'permanent' ? null : Carbon::parse($employee->joining_date)->addYear(),
                'salary' => $salary,
                'payment_frequency' => $paymentFrequency,
                'hourly_rate' => $hourlyRate,
                'daily_rate' => $dailyRate,
                'overtime_rate_multiplier' => 1.5,
                'overtime_eligible' => in_array($employmentType, ['permanent', 'contract']),
                'notice_period_days' => match($employmentType) {
                    'permanent' => 30,
                    'contract' => 14,
                    default => 7,
                },
                'eligible_allowances' => ['house', 'transport', 'medical'],
                'statutory_deductions_applicable' => !in_array($employmentType, ['casual', 'intern']),
                'withholding_tax_applicable' => in_array($employmentType, ['contract', 'freelance']),
                'withholding_tax_rate' => in_array($employmentType, ['contract', 'freelance']) ? 0.05 : 0,
                'status' => 'active',
                'organization_id' => $employee->organization_id ?? 1,
            ]);
        }
        
        $this->command->info('Contracts created for all employees!');
    }
}