<?php

namespace Database\Seeders;

use App\Models\Organization;
use App\Models\SubscriptionPlan;
use App\Models\Invoice;
use App\Models\Payment;
use App\Services\SubscriptionService;
use App\Services\BillingService;
use Illuminate\Database\Seeder;

class DemoSubscriptionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * This seeder creates demo subscription data for existing organizations.
     * It expects organizations to already exist (created by MultitenancyDataSeeder).
     * 
     * For each organization found:
     * - Creates a trial subscription with the Starter plan
     * - Sets trial period to 14 days
     * 
     * Note: This seeder should be run AFTER MultitenancyDataSeeder to ensure
     * the default organization (Qaya Creatives) exists.
     */
    public function run(): void
    {
        $this->command->info('Creating demo subscription data...');

        $subscriptionService = app(SubscriptionService::class);

        // Get plans
        $starterPlan = SubscriptionPlan::where('slug', 'starter')->first();
        $growthPlan = SubscriptionPlan::where('slug', 'growth')->first();
        $businessPlan = SubscriptionPlan::where('slug', 'business')->first();

        if (!$starterPlan || !$growthPlan || !$businessPlan) {
            $this->command->error('Subscription plans not found. Please run SubscriptionPlanSeeder first.');
            return;
        }

        // Get existing organizations
        $organizations = Organization::all();
        
        if ($organizations->count() === 0) {
            $this->command->error('No organizations found. Please run MultitenancyDataSeeder first to create the default organization.');
            return;
        }

        $this->command->info("Found {$organizations->count()} organization(s)");

        // Create subscriptions for organizations
        foreach ($organizations as $organization) {
            // Skip if already has subscription
            if ($organization->subscription) {
                $this->command->info("Organization {$organization->name} already has subscription");
                continue;
            }

            // Use starter plan for all organizations
            $plan = $starterPlan;
            $status = 'trial'; // Start with trial

            $this->command->info("Creating {$status} subscription for {$organization->name} with {$plan->name} plan");

            try {
                // Create subscription with trial
                $subscription = $subscriptionService->createSubscription(
                    $organization,
                    $plan,
                    'monthly',
                    true // Start with trial
                );

                $this->command->info("  ✓ Created subscription: {$subscription->status} (trial ends: {$subscription->trial_ends_at->format('Y-m-d')})");

            } catch (\Exception $e) {
                $this->command->error("Failed to create subscription for {$organization->name}: {$e->getMessage()}");
            }
        }

        $this->command->info("\n📊 Demo Data Summary:");
        $this->command->info('==================');
        
        $stats = [
            'organizations' => Organization::count(),
            'subscriptions' => \App\Models\OrganizationSubscription::count(),
            'active_subscriptions' => \App\Models\OrganizationSubscription::where('status', 'active')->count(),
            'trial_subscriptions' => \App\Models\OrganizationSubscription::where('status', 'trial')->count(),
            'invoices' => Invoice::count(),
            'paid_invoices' => Invoice::where('status', 'paid')->count(),
            'pending_invoices' => Invoice::where('status', 'pending')->count(),
            'payments' => Payment::count(),
            'completed_payments' => Payment::where('status', 'completed')->count(),
        ];

        foreach ($stats as $key => $value) {
            $this->command->info("  {$key}: {$value}");
        }

        $this->command->info("\n✅ Demo subscription data created successfully!");
    }
}
