<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Organization;
use App\Models\User;
use App\Models\Employee;
use App\Models\Department;
use App\Models\Position;
use App\Models\Attendance;
use App\Models\LeaveRequest;
use App\Models\LeaveBalance;
use App\Models\Payroll;
use App\Models\PerformanceReview;
use App\Models\Holiday;
use App\Models\Announcement;
use App\Models\Contract;
use App\Models\P9Form;
use \App\Models\LeaveType;
use App\Models\EmployeeDocument;
use Illuminate\Support\Facades\Hash;

class MultitenancyDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Check if default organization already exists
        $defaultOrg = Organization::where('slug', 'default')->first();
        
        if (!$defaultOrg) {
            // Create Qaya Creatives for existing data
            $defaultOrg = Organization::create([
                'name' => 'Qaya Creatives',
                'slug' => 'default',
                'status' => 'active',
                'settings' => [],
                'onboarding_step' => 'completed', // Mark onboarding as complete
                'onboarding_completed_at' => now(), // Set completion timestamp
            ]);
        } else {
            // Update existing organization to mark onboarding as complete
            $defaultOrg->update([
                'onboarding_step' => 'completed',
                'onboarding_completed_at' => $defaultOrg->onboarding_completed_at ?? now(),
            ]);
        }

        // Create SystemAdmin user
        $systemAdmin = User::firstOrCreate(
            ['email' => 'systemadmin@example.com'],
            [
                'name' => 'System Administrator',
                'password' => Hash::make('password'),
                'is_system_admin' => true,
                'organization_id' => null, // SystemAdmin doesn't belong to any organization
                'email_verified_at' => now(), // Mark as verified
            ]
        );

        // Assign system_admin role
        if (!$systemAdmin->hasRole('system_admin')) {
            $systemAdmin->assignRole('system_admin');
        }

        // Create or update Organization Admin user for Qaya Creatives (verified by default)
        $orgAdmin = User::updateOrCreate(
            ['email' => 'qayacreatives@gmail.com'],
            [
                'name' => 'Admin User',
                'password' => Hash::make('password'),
                'organization_id' => $defaultOrg->id,
                'email_verified_at' => now(), // Mark as verified - no OTP needed
                'must_verify_email' => false, // Don't require OTP verification
                'first_login_completed' => true, // Mark first login as complete
            ]
        );

        // Assign org_admin role
        if (!$orgAdmin->hasRole('org_admin')) {
            $orgAdmin->assignRole('org_admin');
        }

        // Update existing users to belong to Qaya Creatives and mark as verified
        User::whereNull('organization_id')
            ->where('is_system_admin', false)
            ->where('id', '!=', $orgAdmin->id) // Don't update the org admin we just created
            ->update([
                'organization_id' => $defaultOrg->id,
                'email_verified_at' => now(), // Mark as verified
                'must_verify_email' => false, // Don't require OTP verification
                'first_login_completed' => true, // Mark first login as complete
            ]);

        // Update all existing business records to belong to Qaya Creatives
        $tables = [
            Employee::class,
            Department::class,
            Position::class,
            Attendance::class,
            LeaveRequest::class,
            LeaveBalance::class,
            Payroll::class,
            PerformanceReview::class,
            Holiday::class,
            Announcement::class,
            Contract::class,
            P9Form::class,
            EmployeeDocument::class,
            LeaveType::class,
        ];

        foreach ($tables as $model) {
            if (class_exists($model)) {
                $model::whereNull('organization_id')
                    ->update(['organization_id' => $defaultOrg->id]);
            }
        }

        $this->command->info('✅ Qaya Creatives created and existing data migrated');
        $this->command->info('✅ SystemAdmin user created: systemadmin@example.com / password');
        $this->command->info('✅ Organization Admin created: qayacreatives@gmail.com / password (verified)');
    }
}