<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\TrainingCategory;
use App\Models\TrainingProgram;
use App\Models\TrainingSession;
use App\Models\TrainingEnrollment;
use App\Models\TrainingRecommendation;
use App\Models\LearningPath;
use App\Models\LearningPathStep;
use App\Models\LearningPathEnrollment;
use App\Models\TrainingPerformanceImpact;
use App\Models\Employee;
use App\Models\User;
use App\Models\Organization;
use Carbon\Carbon;

class TrainingTestDataSeeder extends Seeder
{
    public function run(): void
    {
        // Get the first organization
        $organization = Organization::first();
        if (!$organization) {
            $this->command->error('No organization found. Please create an organization first.');
            return;
        }

        // Get some users and employees
        $users = User::where('organization_id', $organization->id)->limit(5)->get();
        $employees = Employee::where('organization_id', $organization->id)->limit(5)->get();
        
        if ($users->isEmpty() || $employees->isEmpty()) {
            $this->command->error('No users or employees found. Please create some users and employees first.');
            return;
        }

        $adminUser = $users->first();

        // Create Training Categories
        $categories = [
            ['name' => 'Technical Skills', 'description' => 'Programming, software, and technical competencies', 'color' => '#007bff'],
            ['name' => 'Soft Skills', 'description' => 'Communication, leadership, and interpersonal skills', 'color' => '#28a745'],
            ['name' => 'Compliance', 'description' => 'Regulatory and compliance training', 'color' => '#dc3545'],
            ['name' => 'Safety', 'description' => 'Workplace safety and health training', 'color' => '#ffc107'],
        ];

        foreach ($categories as $categoryData) {
            TrainingCategory::firstOrCreate(
                ['name' => $categoryData['name'], 'organization_id' => $organization->id],
                array_merge($categoryData, ['organization_id' => $organization->id])
            );
        }

        $createdCategories = TrainingCategory::where('organization_id', $organization->id)->get();

        // Create Training Programs
        $programs = [
            [
                'title' => 'Laravel Development Fundamentals',
                'description' => 'Learn the basics of Laravel framework development',
                'type' => 'online',
                'delivery_method' => 'self_paced',
                'duration_hours' => 40,
                'cost_per_participant' => 299.99,
                'difficulty_level' => 'beginner',
                'learning_objectives' => ['Understand MVC architecture', 'Build basic web applications', 'Work with databases'],
            ],
            [
                'title' => 'Leadership Excellence',
                'description' => 'Develop essential leadership and management skills',
                'type' => 'workshop',
                'delivery_method' => 'in_person',
                'duration_hours' => 16,
                'cost_per_participant' => 599.99,
                'difficulty_level' => 'intermediate',
                'learning_objectives' => ['Effective communication', 'Team management', 'Decision making'],
            ],
            [
                'title' => 'Data Protection & GDPR Compliance',
                'description' => 'Understanding data protection laws and compliance requirements',
                'type' => 'online',
                'delivery_method' => 'virtual',
                'duration_hours' => 8,
                'cost_per_participant' => 149.99,
                'difficulty_level' => 'beginner',
                'is_mandatory' => true,
                'learning_objectives' => ['GDPR principles', 'Data handling procedures', 'Compliance requirements'],
            ],
            [
                'title' => 'Advanced Project Management',
                'description' => 'Master advanced project management techniques and methodologies',
                'type' => 'certification',
                'delivery_method' => 'hybrid',
                'duration_hours' => 60,
                'cost_per_participant' => 899.99,
                'difficulty_level' => 'advanced',
                'learning_objectives' => ['Agile methodologies', 'Risk management', 'Stakeholder communication'],
            ],
        ];

        $createdPrograms = collect();
        foreach ($programs as $index => $programData) {
            $category = $createdCategories->get($index % $createdCategories->count());
            
            $program = TrainingProgram::firstOrCreate(
                ['title' => $programData['title'], 'organization_id' => $organization->id],
                array_merge($programData, [
                    'training_category_id' => $category->id,
                    'created_by' => $adminUser->id,
                    'organization_id' => $organization->id,
                ])
            );
            
            $createdPrograms->push($program);
        }

        // Create Training Sessions
        foreach ($createdPrograms as $program) {
            for ($i = 1; $i <= 2; $i++) {
                $startDate = Carbon::now()->addDays(rand(7, 60));
                $endDate = $startDate->copy()->addHours($program->duration_hours);
                
                TrainingSession::firstOrCreate([
                    'training_program_id' => $program->id,
                    'session_code' => $program->id . '-' . $i . '-' . date('Y'),
                    'organization_id' => $organization->id,
                ], [
                    'start_datetime' => $startDate,
                    'end_datetime' => $endDate,
                    'location' => $program->delivery_method === 'in_person' ? 'Conference Room A' : null,
                    'virtual_meeting_link' => $program->delivery_method !== 'in_person' ? 'https://meet.example.com/session-' . $i : null,
                    'max_participants' => rand(10, 30),
                    'status' => 'scheduled',
                    'created_by' => $adminUser->id,
                ]);
            }
        }

        // Create some Training Enrollments
        $sessions = TrainingSession::where('organization_id', $organization->id)->get();
        foreach ($employees->take(3) as $employee) {
            foreach ($sessions->take(2) as $session) {
                TrainingEnrollment::firstOrCreate([
                    'training_session_id' => $session->id,
                    'employee_id' => $employee->id,
                ], [
                    'organization_id' => $organization->id,
                    'enrollment_type' => 'voluntary',
                    'status' => collect(['enrolled', 'completed', 'attended'])->random(),
                    'enrolled_at' => Carbon::now()->subDays(rand(1, 30)),
                    'completion_score' => rand(70, 100),
                    'rating' => rand(3, 5),
                    'enrolled_by' => $adminUser->id,
                ]);
            }
        }

        // Create Training Recommendations
        foreach ($employees->take(3) as $employee) {
            foreach ($createdPrograms->take(2) as $program) {
                TrainingRecommendation::firstOrCreate([
                    'employee_id' => $employee->id,
                    'training_program_id' => $program->id,
                ], [
                    'organization_id' => $organization->id,
                    'recommendation_type' => collect(['skill_gap', 'performance_improvement', 'career_development'])->random(),
                    'reason' => 'Recommended based on performance review and skill assessment',
                    'priority_score' => rand(50, 90),
                    'status' => collect(['pending', 'accepted', 'declined'])->random(),
                    'recommended_at' => Carbon::now()->subDays(rand(1, 15)),
                    'recommended_by' => $adminUser->id,
                ]);
            }
        }

        // Create Learning Paths
        $learningPaths = [
            [
                'title' => 'Full Stack Developer Path',
                'description' => 'Complete journey from beginner to full stack developer',
                'difficulty_level' => 'intermediate',
                'estimated_duration_hours' => 120,
                'learning_outcomes' => ['Frontend development', 'Backend development', 'Database management', 'Deployment'],
            ],
            [
                'title' => 'Leadership Development Track',
                'description' => 'Comprehensive leadership development program',
                'difficulty_level' => 'advanced',
                'estimated_duration_hours' => 80,
                'learning_outcomes' => ['Team leadership', 'Strategic thinking', 'Communication skills'],
            ],
        ];

        foreach ($learningPaths as $pathData) {
            $path = LearningPath::firstOrCreate(
                ['title' => $pathData['title'], 'organization_id' => $organization->id],
                array_merge($pathData, [
                    'created_by' => $adminUser->id,
                    'organization_id' => $organization->id,
                ])
            );

            // Add steps to learning path
            foreach ($createdPrograms->take(3) as $index => $program) {
                LearningPathStep::firstOrCreate([
                    'learning_path_id' => $path->id,
                    'training_program_id' => $program->id,
                ], [
                    'organization_id' => $organization->id,
                    'step_order' => $index + 1,
                    'is_required' => $index < 2, // First 2 steps are required
                ]);
            }

            // Create enrollments
            foreach ($employees->take(2) as $employee) {
                LearningPathEnrollment::firstOrCreate([
                    'learning_path_id' => $path->id,
                    'employee_id' => $employee->id,
                ], [
                    'organization_id' => $organization->id,
                    'status' => collect(['enrolled', 'in_progress', 'completed'])->random(),
                    'enrolled_at' => Carbon::now()->subDays(rand(1, 30)),
                    'progress_percentage' => rand(0, 100),
                    'current_step' => rand(1, 3),
                    'enrolled_by' => $adminUser->id,
                ]);
            }
        }

        // Create Training Performance Impacts
        $completedEnrollments = TrainingEnrollment::where('status', 'completed')->get();
        foreach ($completedEnrollments->take(5) as $enrollment) {
            $beforeScore = rand(20, 35) / 10; // 2.0 - 3.5
            $afterScore = $beforeScore + (rand(5, 15) / 10); // Improvement of 0.5 - 1.5
            
            TrainingPerformanceImpact::firstOrCreate([
                'training_program_id' => $enrollment->session->training_program_id,
                'employee_id' => $enrollment->employee_id,
            ], [
                'organization_id' => $organization->id,
                'performance_score_before' => $beforeScore,
                'performance_score_after' => min(5.0, $afterScore),
                'training_completed_date' => Carbon::now()->subDays(rand(30, 90)),
                'performance_measured_date' => Carbon::now()->subDays(rand(1, 29)),
                'impact_notes' => 'Significant improvement observed in relevant skills and performance metrics',
                'measured_by' => $adminUser->id,
            ]);
        }

        $this->command->info('Training test data created successfully!');
        $this->command->info('Created:');
        $this->command->info('- ' . TrainingCategory::count() . ' Training Categories');
        $this->command->info('- ' . TrainingProgram::count() . ' Training Programs');
        $this->command->info('- ' . TrainingSession::count() . ' Training Sessions');
        $this->command->info('- ' . TrainingEnrollment::count() . ' Training Enrollments');
        $this->command->info('- ' . TrainingRecommendation::count() . ' Training Recommendations');
        $this->command->info('- ' . LearningPath::count() . ' Learning Paths');
        $this->command->info('- ' . TrainingPerformanceImpact::count() . ' Performance Impact Records');
    }
}