# Component Quick Reference Guide

Quick reference for using the new Blade components in the HR application.

---

## Layout Components

### Page Header
Standardized page header with title, description, and action buttons.

```blade
<x-page-header 
    title="Page Title" 
    description="Optional description"
    icon="optional-icon-name">
    <x-slot:actions>
        <a href="#" class="btn btn-primary">Button</a>
    </x-slot:actions>
</x-page-header>
```

**Props:**
- `title` (required) - Page title
- `description` (optional) - Subtitle/description
- `icon` (optional) - Bootstrap icon name

**Slots:**
- `actions` - Action buttons (right side)

---

### Breadcrumb
Consistent breadcrumb navigation.

```blade
<x-breadcrumb :items="[
    ['label' => 'Dashboard', 'url' => route('dashboard')],
    ['label' => 'Users', 'url' => route('users.index')],
    ['label' => 'Create']
]" />
```

**Props:**
- `items` (array) - Array of breadcrumb items
  - `label` (required) - Display text
  - `url` (optional) - Link URL (omit for last item)

---

### Alert
Success/error/warning/info alerts with auto-dismissal.

```blade
<x-alert type="success" :message="session('success')" />
<x-alert type="danger" message="Error occurred" />
<x-alert type="warning" :dismissible="false">Custom content</x-alert>
```

**Props:**
- `type` (default: 'success') - success, danger, warning, info
- `message` (optional) - Alert message
- `dismissible` (default: true) - Show close button
- `icon` (optional) - Custom icon name

---

## Data Display Components

### Empty State
Consistent empty state messaging.

```blade
<x-empty-state 
    icon="inbox" 
    title="No items found"
    description="Optional description">
    <x-slot:actions>
        <a href="#" class="btn btn-primary">Create First Item</a>
    </x-slot:actions>
</x-empty-state>
```

**Props:**
- `icon` (required) - Bootstrap icon name
- `title` (required) - Main message
- `description` (optional) - Additional text

**Slots:**
- `actions` - Action buttons

---

### Status Badge
Auto-detecting status badges with color mapping.

```blade
<x-status-badge :status="$item->status" />
<x-status-badge status="Active" type="success" size="lg" />
```

**Props:**
- `status` (required) - Status value
- `type` (default: 'auto') - auto, success, danger, warning, info, primary, secondary
- `size` (default: 'md') - sm, md, lg

**Auto-Detected Colors:**
- active, approved, completed, published, paid, success → green
- pending, draft, processing → yellow
- rejected, failed, error, cancelled → red
- inactive, archived → gray
- urgent → red
- high → yellow
- medium → blue
- low → gray

---

### User Avatar
User avatars with photo or initials fallback.

```blade
<x-user-avatar :user="$employee" size="md" />
<x-user-avatar :user="$user" size="lg" show-name />
<x-user-avatar :user="$employee" size="xl" show-name show-details />
```

**Props:**
- `user` (required) - User/Employee model
- `size` (default: 'md') - sm (32px), md (40px), lg (64px), xl (100px)
- `showName` (default: false) - Display name next to avatar
- `showDetails` (default: false) - Display email/employee code

---

### Filter Form
Standardized filter forms for index pages.

```blade
<x-filter-form :action="route('users.index')">
    <div class="col-md-4">
        <input type="text" name="search" class="form-control" placeholder="Search...">
    </div>
    <div class="col-md-3">
        <select name="status" class="form-select">
            <option value="">All Status</option>
            <option value="active">Active</option>
        </select>
    </div>
</x-filter-form>
```

**Props:**
- `action` (required) - Form action URL
- `method` (default: 'GET') - HTTP method

**Note:** Filter and Clear buttons are automatically added.

---

### Action Buttons
Reusable action button groups (view/edit/delete).

```blade
<x-action-buttons 
    :item="$user" 
    resource="users"
    :can-view="auth()->user()->can('view', $user)"
    :can-edit="auth()->user()->can('update', $user)"
    :can-delete="auth()->user()->can('delete', $user)">
    <x-slot:extra>
        <button class="btn btn-sm btn-outline-info">Custom</button>
    </x-slot:extra>
</x-action-buttons>
```

**Props:**
- `item` (required) - Model instance
- `resource` (required) - Resource name for routes
- `canView` (default: true) - Show view button
- `canEdit` (default: true) - Show edit button
- `canDelete` (default: true) - Show delete button
- `viewRoute` (optional) - Custom view route
- `editRoute` (optional) - Custom edit route
- `deleteRoute` (optional) - Custom delete route
- `deleteConfirm` (optional) - Custom delete confirmation message
- `size` (default: 'sm') - Button size

**Slots:**
- `extra` - Additional custom buttons

---

## Form Components

### Input
Text input with validation.

```blade
<x-form.input 
    name="title" 
    label="Title" 
    type="text"
    required 
    placeholder="Enter title"
    help="Optional help text" />
```

**Props:**
- `name` (required) - Input name
- `label` (optional) - Field label
- `type` (default: 'text') - Input type
- `required` (default: false) - Required field
- `placeholder` (optional) - Placeholder text
- `value` (optional) - Default value (auto-uses old())
- `help` (optional) - Help text below field

---

### Textarea
Textarea with validation.

```blade
<x-form.textarea 
    name="description" 
    label="Description" 
    rows="4"
    required
    placeholder="Enter description"
    help="Optional help text" />
```

**Props:**
- `name` (required) - Input name
- `label` (optional) - Field label
- `rows` (default: 3) - Number of rows
- `required` (default: false) - Required field
- `placeholder` (optional) - Placeholder text
- `value` (optional) - Default value (auto-uses old())
- `help` (optional) - Help text below field

---

### Select
Select dropdown with validation.

```blade
<x-form.select 
    name="status" 
    label="Status" 
    required
    placeholder="Select status"
    :options="[
        'active' => 'Active',
        'inactive' => 'Inactive'
    ]"
    help="Optional help text" />
```

**Props:**
- `name` (required) - Input name
- `label` (optional) - Field label
- `required` (default: false) - Required field
- `options` (required) - Array of value => label pairs
- `value` (optional) - Selected value (auto-uses old())
- `placeholder` (optional) - Empty option text
- `help` (optional) - Help text below field

---

### Checkbox
Checkbox with validation.

```blade
<x-form.checkbox 
    name="is_active" 
    label="Active"
    :checked="true"
    value="1"
    help="Optional help text" />
```

**Props:**
- `name` (required) - Input name
- `label` (required) - Checkbox label
- `checked` (default: false) - Checked state
- `value` (default: '1') - Checkbox value
- `help` (optional) - Help text below field

---

### Date
Date picker with validation.

```blade
<x-form.date 
    name="start_date" 
    label="Start Date" 
    required
    :value="now()->format('Y-m-d')"
    min="2024-01-01"
    max="2024-12-31"
    help="Optional help text" />
```

**Props:**
- `name` (required) - Input name
- `label` (optional) - Field label
- `required` (default: false) - Required field
- `value` (optional) - Default value (auto-uses old())
- `min` (optional) - Minimum date
- `max` (optional) - Maximum date
- `help` (optional) - Help text below field

---

### Form Actions / Button Group
Form submit/cancel buttons with flexible layout.

```blade
<!-- Using form.actions (simple) -->
<x-form.actions 
    submit-text="Create User"
    :cancel-url="route('users.index')"
    submit-icon="check-circle"
    cancel-icon="x-circle"
    submit-class="btn-primary"
    cancel-class="btn-secondary" />

<!-- Using form.button-group (with alignment options) -->
<x-form.button-group 
    cancelUrl="{{ route('users.index') }}"
    submitText="Create User"
    submitIcon="bi-check-circle"
    cancelIcon="bi-x-circle"
    submitClass="btn-primary"
    alignment="end" />
```

**Props (form.actions):**
- `submitText` (default: 'Save') - Submit button text
- `cancelUrl` (required) - Cancel button URL
- `submitIcon` (default: 'check-circle') - Submit button icon
- `cancelIcon` (default: 'x-circle') - Cancel button icon
- `submitClass` (default: 'btn-primary') - Submit button class
- `cancelClass` (default: 'btn-secondary') - Cancel button class

**Props (form.button-group):**
- `cancelUrl` (optional) - Cancel button URL
- `cancelText` (default: 'Cancel') - Cancel button text
- `submitText` (default: 'Submit') - Submit button text
- `submitIcon` (default: 'bi-check-circle') - Submit button icon
- `cancelIcon` (default: 'bi-x-circle') - Cancel button icon
- `submitClass` (default: 'btn-primary') - Submit button class
- `alignment` (default: 'end') - Flexbox alignment: 'start', 'center', 'end', 'between'

**Slots:**
- `extra` (form.actions) - Additional buttons
- Default slot (form.button-group) - Additional buttons

---

## Common Patterns

### Index Page Pattern
```blade
@extends('layouts.app')

@section('content')
    <x-breadcrumb :items="[...]" />
    
    <x-page-header title="..." description="...">
        <x-slot:actions>
            <a href="#" class="btn btn-primary">New Item</a>
        </x-slot:actions>
    </x-page-header>
    
    <x-alert type="success" :message="session('success')" />
    
    <x-filter-form :action="route('...')">
        <!-- Filter fields -->
    </x-filter-form>
    
    <div class="table-card">
        <table class="table">
            <!-- Table content -->
            @forelse($items as $item)
                <!-- Row -->
            @empty
                <tr>
                    <td colspan="X">
                        <x-empty-state icon="..." title="...">
                            <x-slot:actions>
                                <a href="#" class="btn btn-primary">Create First</a>
                            </x-slot:actions>
                        </x-empty-state>
                    </td>
                </tr>
            @endforelse
        </table>
    </div>
@endsection
```

### Create/Edit Page Pattern
```blade
@extends('layouts.app')

@section('content')
    <x-breadcrumb :items="[...]" />
    
    <div class="table-card">
        <form action="..." method="POST">
            @csrf
            
            <x-form.input name="..." label="..." required />
            <x-form.textarea name="..." label="..." />
            <x-form.select name="..." label="..." :options="[...]" />
            <x-form.date name="..." label="..." />
            <x-form.checkbox name="..." label="..." />
            
            <x-form.actions 
                submit-text="Save"
                :cancel-url="route('...')" />
        </form>
    </div>
@endsection
```

---

## Tips & Best Practices

1. **Always use components for consistency** - Don't mix old and new patterns
2. **Use auto-detecting status badges** - Let the component handle colors
3. **Leverage form components** - They handle validation errors automatically
4. **Use breadcrumbs on all pages** - Improves navigation
5. **Add help text to complex fields** - Improves UX
6. **Use empty states** - Better than "No records found"
7. **Keep action buttons consistent** - Use action-buttons component

---

## Migration Checklist

When refactoring a view:

- [ ] Replace breadcrumb HTML with `<x-breadcrumb>`
- [ ] Replace page header with `<x-page-header>`
- [ ] Replace alerts with `<x-alert>`
- [ ] Replace filter forms with `<x-filter-form>`
- [ ] Replace status badges with `<x-status-badge>`
- [ ] Replace empty states with `<x-empty-state>`
- [ ] Replace form fields with `<x-form.*>` components
- [ ] Replace action buttons with `<x-action-buttons>`
- [ ] Test the refactored view
- [ ] Check for validation errors
- [ ] Verify permissions still work

---

**Last Updated:** February 9, 2026
