@extends('layouts.app')

@section('title', 'Announcements')
@section('page-title', 'Announcements Management')

@section('content')
    <x-breadcrumb :items="[
        ['label' => 'Dashboard', 'url' => route('dashboard')],
        ['label' => 'Announcements']
    ]" />

    <x-page-header 
        title="Announcements" 
        description="Manage company announcements and communications">
        <x-slot:actions>
            @can('create', App\Models\Announcement::class)
            <a href="{{ route('announcements.create') }}" class="btn btn-primary">
                <i class="bi bi-plus-circle"></i> New Announcement
            </a>
            @endcan
        </x-slot:actions>
    </x-page-header>

    <x-alert type="success" :message="session('success')" />

    <!-- Statistics Cards -->
    <x-stats-grid :stats="[
        ['value' => $stats['total'], 'label' => 'Total Announcements', 'icon' => 'megaphone', 'type' => 'primary'],
        ['value' => $stats['published'], 'label' => 'Published', 'icon' => 'check-circle', 'type' => 'success'],
        ['value' => $stats['draft'], 'label' => 'Drafts', 'icon' => 'file-earmark-text', 'type' => 'warning'],
        ['value' => $stats['archived'], 'label' => 'Archived', 'icon' => 'archive', 'type' => 'secondary'],
    ]" />

    <!-- Quick Actions -->
    @can('manage_announcements')
    <div class="row mb-4">
        <div class="col-md-12">
            <div class="table-card bg-light">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="mb-1">Quick Actions</h6>
                        <small class="text-muted">Common announcement management tasks</small>
                    </div>
                    <div class="btn-group">
                        @can('create', App\Models\Announcement::class)
                        <a href="{{ route('announcements.create') }}" class="btn btn-primary btn-sm">
                            <i class="bi bi-plus-circle"></i> New Announcement
                        </a>
                        @endcan
                        <a href="{{ route('announcements.index', ['status' => 'draft']) }}" class="btn btn-outline-warning btn-sm">
                            <i class="bi bi-file-earmark-text"></i> View Drafts
                        </a>
                        <a href="{{ route('announcements.index', ['priority' => 'urgent']) }}" class="btn btn-outline-danger btn-sm">
                            <i class="bi bi-exclamation-triangle"></i> Urgent Items
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
    @endcan

    <x-filter-form :action="route('announcements.index')">
        <div class="col-md-3">
            <label class="form-label">Search</label>
            <input type="text" class="form-control" name="search" 
                   value="{{ request('search') }}" placeholder="Search announcements...">
        </div>
        <div class="col-md-2">
            <label class="form-label">Status</label>
            <select class="form-select" name="status">
                <option value="">All Status</option>
                @can('manage_announcements')
                <option value="draft" {{ request('status') == 'draft' ? 'selected' : '' }}>Draft</option>
                @endcan
                <option value="published" {{ request('status') == 'published' ? 'selected' : '' }}>Published</option>
                @can('manage_announcements')
                <option value="archived" {{ request('status') == 'archived' ? 'selected' : '' }}>Archived</option>
                @endcan
            </select>
        </div>
        <div class="col-md-2">
            <label class="form-label">Priority</label>
            <select class="form-select" name="priority">
                <option value="">All Priorities</option>
                <option value="low" {{ request('priority') == 'low' ? 'selected' : '' }}>Low</option>
                <option value="medium" {{ request('priority') == 'medium' ? 'selected' : '' }}>Medium</option>
                <option value="high" {{ request('priority') == 'high' ? 'selected' : '' }}>High</option>
                <option value="urgent" {{ request('priority') == 'urgent' ? 'selected' : '' }}>Urgent</option>
            </select>
        </div>
    </x-filter-form>

    <div class="row">
        <div class="col-md-12">
            <div class="table-card">

                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Title</th>
                                <th>Status</th>
                                <th>Priority</th>
                                <th>Created By</th>
                                <th>Date Range</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($announcements as $announcement)
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            @if($announcement->is_pinned)
                                                <i class="bi bi-pin-fill text-danger me-2" title="Pinned"></i>
                                            @endif
                                            <div>
                                                @can('view', $announcement)
                                                <a href="{{ route('announcements.show', $announcement) }}" class="fw-bold text-decoration-none">
                                                    {{ $announcement->title }}
                                                </a>
                                                @else
                                                <span class="fw-bold">{{ $announcement->title }}</span>
                                                @endcan
                                                <br>
                                                <small class="text-muted">{{ $announcement->excerpt }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <x-status-badge :status="$announcement->status" />
                                        @if(!$announcement->active)
                                            <x-status-badge status="Inactive" type="danger" class="ms-1" />
                                        @endif
                                    </td>
                                    <td>
                                        <x-status-badge :status="$announcement->priority" />
                                    </td>
                                    <td>
                                        {{ $announcement->creator->name ?? 'N/A' }}
                                        <br>
                                        <small class="text-muted">{{ $announcement->created_at->format('M d, Y') }}</small>
                                    </td>
                                    <td>
                                        <small>
                                            <strong>Start:</strong> {{ $announcement->start_date->format('M d, Y') }}<br>
                                            <strong>End:</strong> {{ $announcement->end_date ? $announcement->end_date->format('M d, Y') : 'No end date' }}
                                        </small>
                                    </td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            @can('view', $announcement)
                                            <a href="{{ route('announcements.show', $announcement) }}" class="btn btn-outline-primary" title="View">
                                                <i class="bi bi-eye"></i>
                                            </a>
                                            @endcan
                                            @can('update', $announcement)
                                            <a href="{{ route('announcements.edit', $announcement) }}" class="btn btn-outline-secondary" title="Edit">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            @endcan
                                            @can('update', $announcement)
                                            <div class="btn-group btn-group-sm">
                                                <button type="button" class="btn btn-outline-info dropdown-toggle" data-bs-toggle="dropdown" title="Actions">
                                                    <i class="bi bi-three-dots"></i>
                                                </button>
                                                <ul class="dropdown-menu">
                                                    <li>
                                                        <form action="{{ route('announcements.toggle-pin', $announcement) }}" method="POST" class="d-inline">
                                                            @csrf
                                                            <button type="submit" class="dropdown-item">
                                                                <i class="bi bi-pin{{ $announcement->is_pinned ? '-fill' : '' }} me-2"></i>
                                                                {{ $announcement->is_pinned ? 'Unpin' : 'Pin' }}
                                                            </button>
                                                        </form>
                                                    </li>
                                                    @if($announcement->status == 'draft')
                                                    <li>
                                                        <form action="{{ route('announcements.publish', $announcement) }}" method="POST" class="d-inline">
                                                            @csrf
                                                            <button type="submit" class="dropdown-item">
                                                                <i class="bi bi-check-circle me-2"></i>Publish
                                                            </button>
                                                        </form>
                                                    </li>
                                                    @endif
                                                    @if($announcement->status == 'published')
                                                    <li>
                                                        <form action="{{ route('announcements.archive', $announcement) }}" method="POST" class="d-inline">
                                                            @csrf
                                                            <button type="submit" class="dropdown-item">
                                                                <i class="bi bi-archive me-2"></i>Archive
                                                            </button>
                                                        </form>
                                                    </li>
                                                    @endif
                                                </ul>
                                            </div>
                                            @endcan
                                            @can('delete', $announcement)
                                            <form action="{{ route('announcements.destroy', $announcement) }}" method="POST" class="d-inline">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" class="btn btn-outline-danger" 
                                                        onclick="return confirm('Are you sure you want to delete this announcement?')" title="Delete">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                            @endcan
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="6">
                                        <x-empty-state 
                                            icon="megaphone" 
                                            title="No announcements found"
                                            description="No announcements match your current filters">
                                            <x-slot:actions>
                                                @can('create', App\Models\Announcement::class)
                                                <a href="{{ route('announcements.create') }}" class="btn btn-primary">
                                                    <i class="bi bi-plus-circle"></i> Create your first announcement
                                                </a>
                                                @endcan
                                            </x-slot:actions>
                                        </x-empty-state>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                @if($announcements->hasPages())
                    <div class="d-flex justify-content-center mt-4">
                        {{ $announcements->links() }}
                    </div>
                @endif
            </div>
        </div>
    </div>
@endsection

@push('styles')
<style>
.announcement-priority-urgent {
    border-left: 4px solid #dc3545;
}
.announcement-priority-high {
    border-left: 4px solid #ffc107;
}
.announcement-priority-medium {
    border-left: 4px solid #0dcaf0;
}
.announcement-priority-low {
    border-left: 4px solid #6c757d;
}

.table-card.bg-light {
    background-color: #f8f9fa !important;
    border: 1px solid #e9ecef;
}

.stat-card:hover {
    transform: translateY(-2px);
    transition: transform 0.2s ease-in-out;
}

.list-group-item:hover {
    background-color: #f8f9fa;
}

.badge {
    font-size: 0.75em;
}

.dropdown-menu {
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
}
</style>
@endpush