@extends('layouts.app')

@section('title', 'Loan Details - ' . $loan->loan_number)

@section('content')
<x-page-header title="Loan Details">
    <x-slot:breadcrumbs>
        <x-breadcrumb :items="[
            ['label' => 'Dashboard', 'url' => route('dashboard')],
            ['label' => 'Loans', 'url' => route('loans.index')],
            ['label' => $loan->loan_number, 'active' => true]
        ]" />
    </x-slot:breadcrumbs>
    
    <x-slot:actions>
        <a href="{{ route('loans.index') }}" class="btn btn-outline-secondary">
            <i class="bi bi-arrow-left"></i> Back to Loans
        </a>
        
        @can('manage_payroll')
            @if($loan->status === 'pending')
                <a href="{{ route('loans.edit', $loan) }}" class="btn btn-outline-primary">
                    <i class="bi bi-pencil"></i> Edit
                </a>
                <button type="button" class="btn btn-success" data-bs-toggle="modal" data-bs-target="#approveModal">
                    <i class="bi bi-check-circle"></i> Approve
                </button>
                <button type="button" class="btn btn-danger" data-bs-toggle="modal" data-bs-target="#rejectModal">
                    <i class="bi bi-x-circle"></i> Reject
                </button>
            @endif
            
            @if($loan->status === 'approved')
                <form action="{{ route('loans.disburse', $loan) }}" method="POST" class="d-inline">
                    @csrf
                    <button type="submit" class="btn btn-primary" onclick="return confirm('Mark this loan as disbursed?')">
                        <i class="bi bi-cash-coin"></i> Mark Disbursed
                    </button>
                </form>
            @endif
            
            @if($loan->status === 'disbursed')
                <form action="{{ route('loans.activate', $loan) }}" method="POST" class="d-inline">
                    @csrf
                    <button type="submit" class="btn btn-success" onclick="return confirm('Activate this loan? Deductions will start from next payroll.')">
                        <i class="bi bi-play-circle"></i> Activate
                    </button>
                </form>
            @endif
        @endcan
    </x-slot:actions>
</x-page-header>

            <div class="row">
                <div class="col-md-8">
                    <!-- Loan Information -->
                    <x-info-card title="Loan Information" icon="bi-info-circle">
                        <div class="row">
                            <div class="col-md-6">
                                <x-detail-row label="Loan Number">
                                    <code>{{ $loan->loan_number }}</code>
                                </x-detail-row>
                                <x-detail-row label="Employee" :value="$loan->employee->full_name" />
                                <x-detail-row label="Employee Code" :value="$loan->employee->employee_code" />
                                <x-detail-row label="Loan Type">
                                    <x-badge color="info">{{ ucwords(str_replace('_', ' ', $loan->loan_type)) }}</x-badge>
                                </x-detail-row>
                                <x-detail-row label="Status">
                                    @php
                                        $statusColors = [
                                            'pending' => 'warning',
                                            'approved' => 'info',
                                            'disbursed' => 'primary',
                                            'active' => 'success',
                                            'completed' => 'secondary',
                                            'defaulted' => 'danger',
                                            'cancelled' => 'dark'
                                        ];
                                    @endphp
                                    <x-badge :color="$statusColors[$loan->status] ?? 'secondary'">
                                        {{ ucfirst($loan->status) }}
                                    </x-badge>
                                </x-detail-row>
                            </div>
                            <div class="col-md-6">
                                <x-detail-row label="Disbursement Date" :value="$loan->disbursement_date->format('M d, Y')" />
                                <x-detail-row label="First Deduction" :value="$loan->first_deduction_date->format('M d, Y')" />
                                <x-detail-row label="Expected Completion" :value="$loan->expected_completion_date->format('M d, Y')" />
                                @if($loan->actual_completion_date)
                                <x-detail-row label="Actual Completion" :value="$loan->actual_completion_date->format('M d, Y')" />
                                @endif
                                <x-detail-row label="Repayment Period" :value="$loan->repayment_period_months . ' months'" />
                            </div>
                        </div>

                        @if($loan->purpose)
                        <div class="mt-3">
                            <strong>Purpose:</strong>
                            <p class="text-muted">{{ $loan->purpose }}</p>
                        </div>
                        @endif

                        @if($loan->approval_notes)
                        <div class="mt-3">
                            <strong>Approval Notes:</strong>
                            <p class="text-muted">{{ $loan->approval_notes }}</p>
                            @if($loan->approver)
                                <small class="text-muted">By {{ $loan->approver->name }} on {{ $loan->approved_at->format('M d, Y H:i') }}</small>
                            @endif
                        </div>
                        @endif
                    </x-info-card>

                    <!-- Financial Summary -->
                    <x-info-card title="Financial Summary" icon="bi-cash-stack">
                        <x-stats-grid>
                            <x-stat-card 
                                title="Principal Amount" 
                                :value="'KES ' . number_format($loan->principal_amount, 2)" 
                                color="primary" />
                            <x-stat-card 
                                title="Interest Rate" 
                                :value="$loan->interest_rate . '%'" 
                                subtitle="Annual"
                                color="info" />
                            <x-stat-card 
                                title="Total Amount" 
                                :value="'KES ' . number_format($loan->total_amount, 2)" 
                                color="warning" />
                            <x-stat-card 
                                title="Monthly Deduction" 
                                :value="'KES ' . number_format($loan->monthly_deduction, 2)" 
                                color="success" />
                        </x-stats-grid>

                        <div class="row mt-4">
                            <div class="col-md-4">
                                <div class="text-center">
                                    <h5>KES {{ number_format($loan->amount_paid, 2) }}</h5>
                                    <small class="text-muted">Amount Paid</small>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="text-center">
                                    <h5>KES {{ number_format($loan->balance, 2) }}</h5>
                                    <small class="text-muted">Balance</small>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="text-center">
                                    <h5>{{ $loan->repayments->count() }} / {{ $loan->repayment_period_months }}</h5>
                                    <small class="text-muted">Payments Made</small>
                                </div>
                            </div>
                        </div>

                        <div class="mt-3">
                            <div class="progress" style="height: 25px;">
                                @php
                                    $percentage = $loan->total_amount > 0 ? ($loan->amount_paid / $loan->total_amount) * 100 : 0;
                                @endphp
                                <div class="progress-bar bg-success" role="progressbar" style="width: {{ $percentage }}%">
                                    {{ number_format($percentage, 1) }}%
                                </div>
                            </div>
                        </div>
                    </x-info-card>

                    <!-- Repayment History -->
                    @if($loan->repayments->count() > 0)
                    <x-table-card title="Repayment History">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Amount</th>
                                    <th>Principal</th>
                                    <th>Interest</th>
                                    <th>Method</th>
                                    <th>Payroll</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($loan->repayments as $repayment)
                                <tr>
                                    <td>{{ $repayment->payment_date->format('M d, Y') }}</td>
                                    <td>KES {{ number_format($repayment->amount, 2) }}</td>
                                    <td>KES {{ number_format($repayment->principal_portion, 2) }}</td>
                                    <td>KES {{ number_format($repayment->interest_portion, 2) }}</td>
                                    <td><x-badge color="secondary">{{ ucwords(str_replace('_', ' ', $repayment->payment_method)) }}</x-badge></td>
                                    <td>
                                        @if($repayment->payroll)
                                            <a href="{{ route('payroll.show', $repayment->payroll) }}">{{ $repayment->payroll->payroll_number }}</a>
                                        @else
                                            -
                                        @endif
                                    </td>
                                </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </x-table-card>
                    @endif
                </div>

                <div class="col-md-4">
                    <!-- Quick Stats -->
                    <x-info-sidebar title="Loan Summary">
                        <x-detail-row label="Interest Amount" :value="'KES ' . number_format($loan->total_amount - $loan->principal_amount, 2)" />
                        <x-detail-row label="Payments Made" :value="$loan->repayments->count()" />
                        <x-detail-row label="Remaining Payments" :value="$loan->repayment_period_months - $loan->repayments->count()" />
                        <x-detail-row label="Progress" :value="number_format(($loan->amount_paid / $loan->total_amount) * 100, 1) . '%'" />
                    </x-info-sidebar>

                    @if($loan->terms_conditions)
                    <x-info-card title="Terms & Conditions" class="mt-3">
                        <p class="text-muted small">{{ $loan->terms_conditions }}</p>
                    </x-info-card>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Approve Modal -->
@can('manage_payroll')
@if($loan->status === 'pending')
<div class="modal fade" id="approveModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="{{ route('loans.approve', $loan) }}">
                @csrf
                <div class="modal-header">
                    <h5 class="modal-title">Approve Loan</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="approval_notes" class="form-label">Approval Notes (Optional)</label>
                        <textarea class="form-control" id="approval_notes" name="approval_notes" rows="3" placeholder="Add any notes about this approval..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">Approve Loan</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="rejectModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="{{ route('loans.reject', $loan) }}">
                @csrf
                <div class="modal-header">
                    <h5 class="modal-title">Reject Loan</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="approval_notes" class="form-label">Reason for Rejection <span class="text-danger">*</span></label>
                        <textarea class="form-control" id="approval_notes" name="approval_notes" rows="3" required placeholder="Please provide a reason for rejecting this loan..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Reject Loan</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endif
@endcan
@endsection
