@extends('layouts.app')

@section('title', 'Pay Invoice')

@section('content')
<x-page-header title="Pay Invoice">
    <x-slot:breadcrumbs>
        <x-breadcrumb :items="[
            ['label' => 'Dashboard', 'url' => route('dashboard')],
            ['label' => 'Subscription', 'url' => route('subscription.show')],
            ['label' => 'Pay Invoice', 'active' => true]
        ]" />
    </x-slot:breadcrumbs>
</x-page-header>

<div class="row justify-content-center">
    <div class="col-md-8">
        <x-info-card title="Pay Invoice" icon="bi-credit-card">
            <!-- Invoice Details -->
            <div class="row mb-4">
                <div class="col-md-6">
                    <h5>Invoice Details</h5>
                    <x-detail-row label="Invoice Number" :value="$invoice->invoice_number" />
                    <x-detail-row label="Issue Date" :value="$invoice->issue_date->format('M d, Y')" />
                    <x-detail-row label="Due Date" :value="$invoice->due_date->format('M d, Y')" />
                </div>
                <div class="col-md-6 text-end">
                    <h3 class="text-primary">KES {{ number_format($invoice->total, 2) }}</h3>
                    <p class="text-muted">Amount Due</p>
                </div>
            </div>

            <!-- Line Items -->
            <div class="mb-4">
                <h5>Items</h5>
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Description</th>
                                <th class="text-end">Amount</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($invoice->line_items as $item)
                            <tr>
                                <td>{{ $item['description'] }}</td>
                                <td class="text-end">KES {{ number_format($item['amount'], 2) }}</td>
                            </tr>
                            @endforeach
                        </tbody>
                        <tfoot>
                            <tr>
                                <th>Total</th>
                                <th class="text-end">KES {{ number_format($invoice->total, 2) }}</th>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            </div>

            @if($invoice->isPaid())
                <!-- Paid Status -->
                <x-alert type="success" class="text-center">
                    <i class="bi bi-check-circle fs-1"></i>
                    <h4 class="mt-2">Invoice Paid</h4>
                    <p>This invoice was paid on {{ $invoice->paid_at->format('M d, Y') }}</p>
                </x-alert>
            @else
                <!-- Payment Form -->
                <div class="border-top pt-4">
                    <h5 class="mb-3"><i class="bi bi-phone"></i> Pay with M-Pesa</h5>
                    
                    <form method="POST" action="{{ route('organization.payments.initiate', $invoice) }}" id="payment-form">
                        @csrf
                        <x-form.input 
                            name="phone_number" 
                            label="M-Pesa Phone Number" 
                            type="tel"
                            placeholder="254712345678"
                            help="Enter your M-Pesa registered phone number"
                            required />
                        
                        <div class="d-grid">
                            <button type="submit" class="btn btn-success btn-lg" id="pay-button">
                                <i class="bi bi-phone"></i> Pay KES {{ number_format($invoice->total, 2) }} with M-Pesa
                            </button>
                        </div>
                    </form>
                </div>
            @endif
        </x-info-card>

        <!-- Payment Status (if payment initiated) -->
        @if(session('payment_id'))
        <x-info-card id="payment-status-card" class="mt-4">
            <div class="text-center">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <h5 class="mt-3">Payment Initiated</h5>
                <p>Please check your phone and enter your M-Pesa PIN to complete the payment.</p>
                <p class="text-muted">We'll update this page automatically when payment is received.</p>
            </div>
        </x-info-card>
        @endif
    </div>
</div>

@if(session('payment_id'))
<script>
// Poll for payment status
let paymentId = {{ session('payment_id') }};
let pollInterval;

function checkPaymentStatus() {
    fetch(`/organization/payments/${paymentId}/status`)
        .then(response => response.json())
        .then(data => {
            if (data.is_completed) {
                clearInterval(pollInterval);
                document.getElementById('payment-status-card').innerHTML = `
                    <div class="card-body text-center">
                        <i class="bi bi-check-circle text-success" style="font-size: 3rem;"></i>
                        <h4 class="text-success mt-3">Payment Successful!</h4>
                        <p>Receipt: ${data.mpesa_receipt_number}</p>
                        <p>Completed: ${data.completed_at}</p>
                        <a href="{{ route('subscription.show') }}" class="btn btn-primary">View Subscription</a>
                    </div>
                `;
            } else if (data.is_failed) {
                clearInterval(pollInterval);
                document.getElementById('payment-status-card').innerHTML = `
                    <div class="card-body text-center">
                        <i class="bi bi-x-circle text-danger" style="font-size: 3rem;"></i>
                        <h4 class="text-danger mt-3">Payment Failed</h4>
                        <p>Please try again or contact support.</p>
                        <button onclick="location.reload()" class="btn btn-primary">Try Again</button>
                    </div>
                `;
            }
        })
        .catch(error => {
            console.error('Error checking payment status:', error);
        });
}

// Start polling every 3 seconds
pollInterval = setInterval(checkPaymentStatus, 3000);

// Stop polling after 5 minutes
setTimeout(() => {
    clearInterval(pollInterval);
}, 300000);
</script>
@endif
@endsection
