@extends('layouts.app')

@section('title', 'Bulk Generate P9 Forms')
@section('page-title', 'Bulk Generate P9 Forms')

@section('content')
    <div class="mb-4">
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="{{ route('dashboard') }}">Dashboard</a></li>
                <li class="breadcrumb-item"><a href="{{ route('p9-forms.index') }}">P9 Forms</a></li>
                <li class="breadcrumb-item active">Bulk Generate</li>
            </ol>
        </nav>
    </div>

    <div class="row">
        <div class="col-md-10 mx-auto">
            <div class="table-card">
                <h5 class="mb-4">Generate P9 Forms for Multiple Employees</h5>

                <div class="alert alert-info">
                    <i class="bi bi-info-circle me-2"></i>
                    <strong>Bulk P9 Generation:</strong> Select employees and tax year to generate P9 forms. 
                    Only employees with complete payroll records for the selected year will be processed.
                </div>

                <form action="{{ route('p9-forms.bulk-generate') }}" method="POST">
                    @csrf

                    <div class="row mb-4">
                        <div class="col-md-6">
                            <label for="year" class="form-label">Tax Year <span class="text-danger">*</span></label>
                            <select class="form-select @error('year') is-invalid @enderror"
                                    id="year" name="year" required>
                                @foreach($years as $year)
                                    <option value="{{ $year }}" {{ old('year', now()->year - 1) == $year ? 'selected' : '' }}>
                                        {{ $year }}
                                    </option>
                                @endforeach
                            </select>
                            @error('year')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <small class="text-muted">Select the tax year for P9 form generation</small>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Filter Options</label>
                            <div class="d-flex gap-2">
                                <button type="button" class="btn btn-outline-primary btn-sm" onclick="selectAll()">
                                    <i class="bi bi-check-all"></i> Select All
                                </button>
                                <button type="button" class="btn btn-outline-secondary btn-sm" onclick="selectNone()">
                                    <i class="bi bi-x-square"></i> Select None
                                </button>
                                <button type="button" class="btn btn-outline-success btn-sm" onclick="selectEligible()">
                                    <i class="bi bi-shield-check"></i> Select Eligible Only
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th width="50">
                                        <input type="checkbox" id="select-all" class="form-check-input">
                                    </th>
                                    <th>Employee</th>
                                    <th>Department</th>
                                    <th>KRA PIN</th>
                                    <th>Basic Salary</th>
                                    <th>P9 Status</th>
                                    <th>Eligibility</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($employees as $employee)
                                    <tr class="employee-row" data-eligible="{{ $employee->hasCompleteKraInfo() && $employee->effective_basic_salary > 0 ? 'true' : 'false' }}">
                                        <td>
                                            <input type="checkbox" name="employee_ids[]" value="{{ $employee->id }}" 
                                                   class="form-check-input employee-checkbox">
                                        </td>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                @if($employee->photo)
                                                    <img src="{{ asset('storage/' . $employee->photo) }}"
                                                         alt="{{ $employee->full_name }}"
                                                         class="rounded-circle me-2"
                                                         style="width: 32px; height: 32px; object-fit: cover;">
                                                @else
                                                    <div class="rounded-circle bg-primary text-white d-inline-flex align-items-center justify-content-center me-2"
                                                         style="width: 32px; height: 32px; font-size: 12px;">
                                                        {{ substr($employee->first_name, 0, 1) }}{{ substr($employee->last_name, 0, 1) }}
                                                    </div>
                                                @endif
                                                <div>
                                                    <strong>{{ $employee->full_name }}</strong><br>
                                                    <small class="text-muted">{{ $employee->employee_code }}</small>
                                                </div>
                                            </div>
                                        </td>
                                        <td>{{ $employee->department->name ?? 'N/A' }}</td>
                                        <td>
                                            @if($employee->kra_pin)
                                                <code>{{ $employee->kra_pin }}</code>
                                            @else
                                                <span class="badge bg-warning text-dark">Not Set</span>
                                            @endif
                                        </td>
                                        <td>
                                            @if($employee->effective_basic_salary > 0)
                                                <strong>KES {{ number_format($employee->effective_basic_salary, 2) }}</strong>
                                            @else
                                                <span class="badge bg-warning text-dark">Not Set</span>
                                            @endif
                                        </td>
                                        <td>
                                            @php
                                                $hasP9 = $employee->p9Forms()->where('year', old('year', now()->year - 1))->exists();
                                            @endphp
                                            @if($hasP9)
                                                <span class="badge bg-success">
                                                    <i class="bi bi-check-circle"></i> Generated
                                                </span>
                                            @else
                                                <span class="badge bg-secondary">
                                                    <i class="bi bi-dash-circle"></i> Not Generated
                                                </span>
                                            @endif
                                        </td>
                                        <td>
                                            @if($employee->hasCompleteKraInfo() && $employee->effective_basic_salary > 0)
                                                <span class="badge bg-success" title="Ready for P9 generation">
                                                    <i class="bi bi-check-circle"></i> Eligible
                                                </span>
                                            @else
                                                <span class="badge bg-warning text-dark" title="Missing KRA PIN, NHIF, NSSF, or Basic Salary">
                                                    <i class="bi bi-exclamation-triangle"></i> Incomplete
                                                </span>
                                            @endif
                                        </td>
                                    </tr>
                                @empty
                                    <tr>
                                        <td colspan="7" class="text-center py-4">
                                            <i class="bi bi-people" style="font-size: 48px; color: #ccc;"></i>
                                            <p class="text-muted mt-3 mb-0">No active employees found</p>
                                        </td>
                                    </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>

                    @if($employees->count() > 0)
                        <div class="alert alert-warning mt-4">
                            <h6><i class="bi bi-exclamation-triangle me-2"></i>Important Notes:</h6>
                            <ul class="mb-0">
                                <li>Only employees with complete KRA information and payroll records will be processed</li>
                                <li>Existing P9 forms for the selected year will be updated</li>
                                <li>Employees must have payroll records for the entire selected year</li>
                                <li>Generated P9 forms can be downloaded individually or in bulk</li>
                            </ul>
                        </div>

                        <div class="mt-4">
                            <button type="submit" class="btn btn-success btn-lg" id="generate-btn" disabled>
                                <i class="bi bi-files"></i> Generate P9 Forms for Selected Employees
                            </button>
                            <a href="{{ route('p9-forms.index') }}" class="btn btn-secondary btn-lg">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                        </div>
                    @endif
                </form>
            </div>
        </div>
    </div>
@endsection

@push('scripts')
<script>
    // Handle select all checkbox
    document.getElementById('select-all').addEventListener('change', function() {
        const checkboxes = document.querySelectorAll('.employee-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
        });
        updateGenerateButton();
    });

    // Handle individual checkboxes
    document.querySelectorAll('.employee-checkbox').forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            updateSelectAllCheckbox();
            updateGenerateButton();
        });
    });

    // Select all function
    function selectAll() {
        const checkboxes = document.querySelectorAll('.employee-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = true;
        });
        updateSelectAllCheckbox();
        updateGenerateButton();
    }

    // Select none function
    function selectNone() {
        const checkboxes = document.querySelectorAll('.employee-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = false;
        });
        updateSelectAllCheckbox();
        updateGenerateButton();
    }

    // Select eligible only function
    function selectEligible() {
        const rows = document.querySelectorAll('.employee-row');
        rows.forEach(row => {
            const checkbox = row.querySelector('.employee-checkbox');
            const isEligible = row.getAttribute('data-eligible') === 'true';
            checkbox.checked = isEligible;
        });
        updateSelectAllCheckbox();
        updateGenerateButton();
    }

    // Update select all checkbox state
    function updateSelectAllCheckbox() {
        const checkboxes = document.querySelectorAll('.employee-checkbox');
        const selectAllCheckbox = document.getElementById('select-all');
        const checkedCount = document.querySelectorAll('.employee-checkbox:checked').length;
        
        if (checkedCount === 0) {
            selectAllCheckbox.indeterminate = false;
            selectAllCheckbox.checked = false;
        } else if (checkedCount === checkboxes.length) {
            selectAllCheckbox.indeterminate = false;
            selectAllCheckbox.checked = true;
        } else {
            selectAllCheckbox.indeterminate = true;
        }
    }

    // Update generate button state
    function updateGenerateButton() {
        const checkedCount = document.querySelectorAll('.employee-checkbox:checked').length;
        const generateBtn = document.getElementById('generate-btn');
        
        if (checkedCount > 0) {
            generateBtn.disabled = false;
            generateBtn.innerHTML = `<i class="bi bi-files"></i> Generate P9 Forms for ${checkedCount} Employee${checkedCount > 1 ? 's' : ''}`;
        } else {
            generateBtn.disabled = true;
            generateBtn.innerHTML = '<i class="bi bi-files"></i> Generate P9 Forms for Selected Employees';
        }
    }

    // Initialize button state
    updateGenerateButton();
</script>
@endpush