@extends('layouts.app')

@section('title', 'Bulk Generate Payroll')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1 class="h3 mb-0">Bulk Generate Payroll</h1>
                <a href="{{ route('payroll.index') }}" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left"></i> Back to Payroll
                </a>
            </div>

            @if($errors->any())
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    @foreach($errors->all() as $error)
                        <div>{{ $error }}</div>
                    @endforeach
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            @endif

            <form method="POST" action="{{ route('payroll.bulkStore') }}" id="bulkPayrollForm">
                @csrf
                
                <div class="row">
                    <div class="col-md-8">
                        <!-- Employee Selection -->
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">Select Employees</h5>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="selectAll">
                                        <label class="form-check-label fw-bold" for="selectAll">
                                            Select All Employees ({{ $employees->count() }})
                                        </label>
                                    </div>
                                </div>

                                @if($employees->count() > 0)
                                    <div class="row">
                                        @foreach($employees as $employee)
                                            <div class="col-md-6 mb-3">
                                                <div class="card border">
                                                    <div class="card-body p-3">
                                                        <div class="form-check">
                                                            <input class="form-check-input employee-checkbox" type="checkbox" 
                                                                   name="employee_ids[]" value="{{ $employee->id }}" 
                                                                   id="employee{{ $employee->id }}">
                                                            <label class="form-check-label w-100" for="employee{{ $employee->id }}">
                                                                <div class="d-flex justify-content-between align-items-start">
                                                                    <div>
                                                                        <strong>{{ $employee->full_name }}</strong><br>
                                                                        <small class="text-muted">{{ $employee->employee_code }}</small><br>
                                                                        <small class="text-muted">{{ $employee->department->name ?? 'No Department' }}</small>
                                                                    </div>
                                                                    <div class="text-end">
                                                                        <span class="badge bg-info">{{ ucfirst($employee->activeContract->employment_type) }}</span><br>
                                                                        <small class="text-muted">{{ ucfirst($employee->activeContract->payment_frequency) }}</small>
                                                                    </div>
                                                                </div>
                                                            </label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        @endforeach
                                    </div>
                                @else
                                    <div class="text-center py-4">
                                        <i class="bi bi-users-slash fa-3x text-muted mb-3"></i>
                                        <h5 class="text-muted">No Active Employees Found</h5>
                                        <p class="text-muted">Make sure employees have active contracts before generating payroll.</p>
                                    </div>
                                @endif
                            </div>
                        </div>
                    </div>

                    <div class="col-md-4">
                        <!-- Payroll Settings -->
                        <div class="card">
                            <div class="card-header">
                                <h5 class="card-title mb-0">Payroll Period</h5>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <label for="year" class="form-label">Year</label>
                                    <select name="year" id="year" class="form-select" required>
                                        @foreach($years as $year)
                                            <option value="{{ $year }}" {{ $year == date('Y') ? 'selected' : '' }}>
                                                {{ $year }}
                                            </option>
                                        @endforeach
                                    </select>
                                </div>

                                <div class="mb-3">
                                    <label for="month" class="form-label">Month</label>
                                    <select name="month" id="month" class="form-select" required>
                                        @foreach($months as $num => $name)
                                            <option value="{{ $num }}" {{ $num == date('n') ? 'selected' : '' }}>
                                                {{ $name }}
                                            </option>
                                        @endforeach
                                    </select>
                                </div>

                                <div class="alert alert-info">
                                    <i class="bi bi-info-circle"></i>
                                    <strong>Note:</strong> Payroll will be generated based on:
                                    <ul class="mb-0 mt-2">
                                        <li>Employee contracts and rates</li>
                                        <li>Attendance records for the period</li>
                                        <li>Employment type tax rules</li>
                                        <li>Statutory deductions (PAYE, NHIF, NSSF, Housing Levy)</li>
                                    </ul>
                                </div>

                                <div class="d-grid gap-2">
                                    <button type="submit" class="btn btn-success btn-lg" id="generateBtn" disabled>
                                        <i class="bi bi-cogs"></i> Generate Payroll
                                    </button>
                                    <a href="{{ route('payroll.index') }}" class="btn btn-outline-secondary">
                                        Cancel
                                    </a>
                                </div>
                            </div>
                        </div>

                        <!-- Summary Card -->
                        <div class="card mt-3">
                            <div class="card-header">
                                <h6 class="card-title mb-0">Selection Summary</h6>
                            </div>
                            <div class="card-body">
                                <div class="row text-center">
                                    <div class="col-6">
                                        <div class="border-end">
                                            <h4 class="mb-0" id="selectedCount">0</h4>
                                            <small class="text-muted">Selected</small>
                                        </div>
                                    </div>
                                    <div class="col-6">
                                        <h4 class="mb-0">{{ $employees->count() }}</h4>
                                        <small class="text-muted">Total</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const selectAllCheckbox = document.getElementById('selectAll');
    const employeeCheckboxes = document.querySelectorAll('.employee-checkbox');
    const generateBtn = document.getElementById('generateBtn');
    const selectedCountElement = document.getElementById('selectedCount');

    function updateSelectedCount() {
        const selectedCount = document.querySelectorAll('.employee-checkbox:checked').length;
        selectedCountElement.textContent = selectedCount;
        generateBtn.disabled = selectedCount === 0;
    }

    selectAllCheckbox.addEventListener('change', function() {
        employeeCheckboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
        });
        updateSelectedCount();
    });

    employeeCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            const allChecked = Array.from(employeeCheckboxes).every(cb => cb.checked);
            const noneChecked = Array.from(employeeCheckboxes).every(cb => !cb.checked);
            
            selectAllCheckbox.checked = allChecked;
            selectAllCheckbox.indeterminate = !allChecked && !noneChecked;
            
            updateSelectedCount();
        });
    });

    // Initial count update
    updateSelectedCount();

    // Form submission confirmation
    document.getElementById('bulkPayrollForm').addEventListener('submit', function(e) {
        const selectedCount = document.querySelectorAll('.employee-checkbox:checked').length;
        const year = document.getElementById('year').value;
        const month = document.getElementById('month').options[document.getElementById('month').selectedIndex].text;
        
        if (!confirm(`Are you sure you want to generate payroll for ${selectedCount} employees for ${month} ${year}?`)) {
            e.preventDefault();
        }
    });
});
</script>
@endpush
@endsection