@extends('layouts.app')
@section('title', $jobPosting->title)
@section('page-title', 'Job Posting Details')

@section('content')
<x-breadcrumb :items="[
    ['label' => 'Dashboard', 'url' => route('dashboard')],
    ['label' => 'Recruitment', 'url' => route('job-postings.index')],
    ['label' => $jobPosting->title, 'active' => true]
]" />

<div class="row">
    <div class="col-md-8">
        <!-- Job Details Card -->
        <x-info-card icon="briefcase" class="mb-4">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h5 class="mb-0">{{ $jobPosting->title }}</h5>
                <x-status-badge :status="$jobPosting->status" />
            </div>

            <table class="table table-borderless">
                <x-detail-row label="Department" :value="$jobPosting->department->name" />
                <x-detail-row label="Position" :value="$jobPosting->position->name ?? 'Not specified'" />
                <x-detail-row label="Location" :value="$jobPosting->location" />
                <x-detail-row label="Employment Type" :value="ucfirst($jobPosting->employment_type)" />
                @if($jobPosting->salary_range)
                <x-detail-row label="Salary Range" :value="$jobPosting->salary_range" />
                @endif
                @if($jobPosting->closes_at)
                <x-detail-row label="Application Deadline" :value="$jobPosting->closes_at->format('M d, Y')" />
                @endif
                <x-detail-row label="Created By" :value="$jobPosting->creator->name" />
                <x-detail-row label="Posted" :value="$jobPosting->created_at->format('M d, Y')" />
            </table>

            <!-- Public Link Section -->
            <x-alert type="info" dismissible="false">
                <h6><i class="bi bi-link-45deg"></i> Shareable Application Link</h6>
                <div class="input-group">
                    <input type="text" class="form-control" id="publicLink" value="{{ $jobPosting->public_url }}" readonly>
                    <button class="btn btn-outline-primary" type="button" onclick="copyToClipboard()">
                        <i class="bi bi-clipboard"></i> Copy
                    </button>
                    <a href="{{ $jobPosting->public_url }}" target="_blank" class="btn btn-outline-success">
                        <i class="bi bi-eye"></i> Preview
                    </a>
                </div>
                <small class="text-muted">Share this link via email, WhatsApp, or social media to collect applications.</small>
            </x-alert>

            <div class="mb-3">
                <h6>Job Description</h6>
                <div class="border p-3 rounded bg-light">
                    {!! nl2br(e($jobPosting->description)) !!}
                </div>
            </div>

            <div class="mb-3">
                <h6>Requirements</h6>
                <div class="border p-3 rounded bg-light">
                    {!! nl2br(e($jobPosting->requirements)) !!}
                </div>
            </div>

            <div class="mb-3">
                <h6>Key Responsibilities</h6>
                <div class="border p-3 rounded bg-light">
                    {!! nl2br(e($jobPosting->responsibilities)) !!}
                </div>
            </div>
        </x-info-card>

        <!-- Applications Section -->
        <x-info-card title="Applications ({{ $applicationStats['total'] }})" icon="people">
            @if($jobPosting->applications->count() > 0)
                @foreach($jobPosting->applications as $application)
                <div class="card border mb-3">
                    <div class="card-body">
                        <div class="row align-items-center">
                            <!-- Candidate Info -->
                            <div class="col-md-3">
                                <h6 class="mb-1">{{ $application->full_name }}</h6>
                                <small class="text-muted d-block"><i class="bi bi-envelope"></i> {{ $application->email }}</small>
                                <small class="text-muted d-block"><i class="bi bi-phone"></i> {{ $application->phone }}</small>
                                <small class="text-muted d-block"><i class="bi bi-calendar"></i> {{ $application->applied_at->format('M d, Y') }}</small>
                            </div>

                            <!-- Status Badge -->
                            <div class="col-md-2 text-center">
                                <x-status-badge :status="$application->status" class="fs-6 px-3 py-2" />
                            </div>

                            <!-- Quick Actions -->
                            <div class="col-md-7">
                                <div class="d-flex flex-wrap gap-2">
                                    <!-- CV Actions -->
                                    @if($application->cv_path)
                                        <a href="{{ route('applications.cv.view', $application) }}" target="_blank" class="btn btn-sm btn-outline-primary">
                                            <i class="bi bi-eye"></i> View CV
                                        </a>
                                    @endif

                                    <!-- Message -->
                                    <a href="{{ route('applications.messages', $application) }}" class="btn btn-sm btn-outline-info">
                                        <i class="bi bi-chat-dots"></i> Message
                                    </a>

                                    <!-- Status Actions based on current status -->
                                    @if($application->status === 'new')
                                        <button onclick="updateStatus({{ $application->id }}, 'screening')" class="btn btn-sm btn-warning">
                                            <i class="bi bi-search"></i> Start Screening
                                        </button>
                                        <button onclick="updateStatus({{ $application->id }}, 'shortlisted')" class="btn btn-sm btn-info">
                                            <i class="bi bi-star"></i> Shortlist
                                        </button>
                                        <button onclick="updateStatus({{ $application->id }}, 'rejected')" class="btn btn-sm btn-outline-danger">
                                            <i class="bi bi-x-circle"></i> Reject
                                        </button>
                                    @endif

                                    @if($application->status === 'screening')
                                        <button onclick="updateStatus({{ $application->id }}, 'shortlisted')" class="btn btn-sm btn-info">
                                            <i class="bi bi-star"></i> Shortlist
                                        </button>
                                        <button onclick="updateStatus({{ $application->id }}, 'rejected')" class="btn btn-sm btn-outline-danger">
                                            <i class="bi bi-x-circle"></i> Reject
                                        </button>
                                    @endif

                                    @if($application->status === 'shortlisted')
                                        <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#scheduleInterviewModal{{ $application->id }}">
                                            <i class="bi bi-calendar-event"></i> Schedule Interview
                                        </button>
                                        <button onclick="updateStatus({{ $application->id }}, 'rejected')" class="btn btn-sm btn-outline-danger">
                                            <i class="bi bi-x-circle"></i> Reject
                                        </button>
                                    @endif

                                    @if($application->status === 'interviewed')
                                        <a href="{{ route('applications.offer.create', $application) }}" class="btn btn-sm btn-success">
                                            <i class="bi bi-trophy"></i> Create Offer
                                        </a>
                                        <button onclick="updateStatus({{ $application->id }}, 'rejected')" class="btn btn-sm btn-outline-danger">
                                            <i class="bi bi-x-circle"></i> Reject
                                        </button>
                                    @endif

                                    @if($application->status === 'offered')
                                        @if($application->jobOffer)
                                            <a href="{{ route('job-offers.show', $application->jobOffer) }}" class="btn btn-sm btn-success">
                                                <i class="bi bi-eye"></i> View Offer
                                            </a>
                                        @endif
                                    @endif
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                @endforeach
            @else
                <x-empty-state 
                    icon="inbox"
                    title="No applications yet"
                    description="Share the application link to start receiving applications." />
            @endif
        </x-info-card>
    </div>

    <div class="col-md-4">
        <!-- Statistics Card -->
        <x-info-card title="Application Statistics" icon="bar-chart" class="mb-4">
            <div class="row text-center">
                <div class="col-6 mb-3">
                    <div class="border-end pe-3">
                        <h4 class="text-primary mb-0">{{ $applicationStats['total'] }}</h4>
                        <small class="text-muted">Total</small>
                    </div>
                </div>
                <div class="col-6 mb-3">
                    <div class="ps-3">
                        <h4 class="text-warning mb-0">{{ $applicationStats['new'] ?? 0 }}</h4>
                        <small class="text-muted">New</small>
                    </div>
                </div>
                <div class="col-6">
                    <div class="border-end pe-3">
                        <h4 class="text-info mb-0">{{ $applicationStats['shortlisted'] ?? 0 }}</h4>
                        <small class="text-muted">Shortlisted</small>
                    </div>
                </div>
                <div class="col-6">
                    <div class="ps-3">
                        <h4 class="text-success mb-0">{{ $applicationStats['offered'] ?? 0 }}</h4>
                        <small class="text-muted">Offered</small>
                    </div>
                </div>
            </div>
        </x-info-card>

        <!-- Actions Card -->
        <x-info-card title="Actions" icon="gear">
            <div class="d-grid gap-2">
                @can('edit_job_postings')
                    <a href="{{ route('job-postings.edit', $jobPosting) }}" class="btn btn-outline-primary">
                        <i class="bi bi-pencil"></i> Edit Job Posting
                    </a>
                @endcan

                @if($jobPosting->status === 'active')
                    @can('edit_job_postings')
                        <form method="POST" action="{{ route('job-postings.close', $jobPosting) }}">
                            @csrf
                            <button type="submit" class="btn btn-outline-warning w-100" onclick="return confirm('Are you sure you want to close this job posting?')">
                                <i class="bi bi-x-circle"></i> Close Posting
                            </button>
                        </form>
                    @endcan
                @endif

                @can('delete_job_postings')
                    @if($applicationStats['total'] == 0)
                        <form method="POST" action="{{ route('job-postings.destroy', $jobPosting) }}">
                            @csrf
                            @method('DELETE')
                            <button type="submit" class="btn btn-outline-danger w-100" onclick="return confirm('Are you sure you want to delete this job posting?')">
                                <i class="bi bi-trash"></i> Delete Posting
                            </button>
                        </form>
                    @endif
                @endcan

                <a href="{{ route('job-postings.index') }}" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left"></i> Back to Listings
                </a>
            </div>
        </x-info-card>
    </div>
</div>

<!-- Interview Scheduling Modals -->
@foreach($jobPosting->applications as $application)
<x-modal id="scheduleInterviewModal{{ $application->id }}" title="Schedule Interview - {{ $application->full_name }}">
    <form method="POST" action="{{ route('interviews.store') }}">
        @csrf
        <input type="hidden" name="application_id" value="{{ $application->id }}">
        
        <x-form.input 
            name="scheduled_at" 
            label="Date & Time" 
            type="datetime-local"
            required />
        
        <x-form.input 
            name="location" 
            label="Location (Optional)" 
            placeholder="Office - Room 301" />
        
        <x-form.input 
            name="meeting_link" 
            label="Meeting Link (Optional)" 
            type="url"
            placeholder="https://zoom.us/j/..." />
        
        <x-form.textarea 
            name="notes" 
            label="Notes (Optional)" 
            rows="2" />
        
        <x-slot:footer>
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
            <button type="submit" class="btn btn-primary">Schedule Interview</button>
        </x-slot:footer>
    </form>
</x-modal>
@endforeach
@endsection

@push('scripts')
<script>
function copyToClipboard() {
    const input = document.getElementById('publicLink');
    input.select();
    input.setSelectionRange(0, 99999); // For mobile devices
    
    navigator.clipboard.writeText(input.value).then(function() {
        // Show success message
        const button = event.target.closest('button');
        const originalText = button.innerHTML;
        button.innerHTML = '<i class="bi bi-check"></i> Copied!';
        button.classList.remove('btn-outline-primary');
        button.classList.add('btn-success');
        
        setTimeout(function() {
            button.innerHTML = originalText;
            button.classList.remove('btn-success');
            button.classList.add('btn-outline-primary');
        }, 2000);
    });
}

function updateStatus(applicationId, status) {
    const statusLabels = {
        'screening': 'Screening',
        'shortlisted': 'Shortlisted',
        'interviewed': 'Interviewed',
        'rejected': 'Rejected'
    };
    
    if (!confirm(`Are you sure you want to change the status to ${statusLabels[status]}?`)) {
        return;
    }
    
    // Get CSRF token
    const csrfToken = document.querySelector('meta[name="csrf-token"]');
    if (!csrfToken) {
        alert('Security token not found. Please refresh the page and try again.');
        return;
    }
    
    // Show loading state
    const btn = event.target;
    const originalText = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Updating...';
    
    fetch(`/applications/${applicationId}/update-status`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken.content
        },
        body: JSON.stringify({ status: status })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('Error: ' + (data.message || 'Failed to update status'));
            btn.disabled = false;
            btn.innerHTML = originalText;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Error updating status. Please try again.');
        btn.disabled = false;
        btn.innerHTML = originalText;
    });
}
</script>
@endpush
