@extends('layouts.app')

@section('title', 'Schedule Training Session')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <x-breadcrumb :items="[
                ['label' => 'Training Sessions', 'url' => route('training.sessions.index')],
                ['label' => 'Schedule Session']
            ]" />
            
            <x-page-header title="Schedule Training Session" />

            <form method="POST" action="{{ route('training.sessions.store') }}">
                @csrf
                <div class="row">
                    <div class="col-lg-8">
                        <div class="card mb-4">
                            <div class="card-header">
                                <h5 class="mb-0">Session Details</h5>
                            </div>
                            <div class="card-body">
                                <x-form.select 
                                    name="training_program_id" 
                                    label="Training Program" 
                                    required
                                    :value="old('training_program_id', $selectedProgram?->id)"
                                    :options="$programs->mapWithKeys(fn($p) => [$p->id => $p->title . ' (' . $p->category->name . ')'])->toArray()" />

                                <div class="row">
                                    <div class="col-md-6">
                                        <x-form.input 
                                            type="datetime-local" 
                                            name="start_datetime" 
                                            label="Start Date & Time" 
                                            required />
                                    </div>
                                    <div class="col-md-6">
                                        <x-form.input 
                                            type="datetime-local" 
                                            name="end_datetime" 
                                            label="End Date & Time" 
                                            required />
                                    </div>
                                </div>

                                <x-form.input 
                                    name="location" 
                                    label="Location"
                                    placeholder="e.g., Conference Room A, Building 1" />

                                <x-form.input 
                                    type="url" 
                                    name="virtual_meeting_link" 
                                    label="Virtual Meeting Link"
                                    placeholder="https://zoom.us/j/123456789" />

                                <x-form.textarea name="session_notes" label="Session Notes" rows="4" />
                            </div>
                        </div>
                    </div>

                    <div class="col-lg-4">
                        <div class="card mb-4">
                            <div class="card-header">
                                <h5 class="mb-0">Session Settings</h5>
                            </div>
                            <div class="card-body">
                                <x-form.input 
                                    type="number" 
                                    name="max_participants" 
                                    label="Max Participants"
                                    min="1"
                                    help="Leave empty to use program default" />

                                <x-form.select 
                                    name="instructor_id" 
                                    label="Instructor"
                                    :options="$instructors->pluck('name', 'id')->toArray()" />
                            </div>
                        </div>

                        @if($selectedProgram)
                            <div class="card mb-4">
                                <div class="card-header">
                                    <h5 class="mb-0">Program Information</h5>
                                </div>
                                <div class="card-body">
                                    <div class="row mb-2">
                                        <div class="col-sm-6"><strong>Duration:</strong></div>
                                        <div class="col-sm-6">{{ $selectedProgram->formatted_duration }}</div>
                                    </div>
                                    <div class="row mb-2">
                                        <div class="col-sm-6"><strong>Type:</strong></div>
                                        <div class="col-sm-6">{{ ucfirst($selectedProgram->type) }}</div>
                                    </div>
                                    <div class="row mb-2">
                                        <div class="col-sm-6"><strong>Delivery:</strong></div>
                                        <div class="col-sm-6">{{ ucfirst(str_replace('_', ' ', $selectedProgram->delivery_method)) }}</div>
                                    </div>
                                    @if($selectedProgram->max_participants)
                                        <div class="row mb-2">
                                            <div class="col-sm-6"><strong>Max Participants:</strong></div>
                                            <div class="col-sm-6">{{ $selectedProgram->max_participants }}</div>
                                        </div>
                                    @endif
                                    @if($selectedProgram->cost_per_participant > 0)
                                        <div class="row mb-2">
                                            <div class="col-sm-6"><strong>Cost:</strong></div>
                                            <div class="col-sm-6">Ksh{{ number_format($selectedProgram->cost_per_participant, 2) }}</div>
                                        </div>
                                    @endif
                                </div>
                            </div>
                        @endif

                        <div class="card">
                            <div class="card-body">
                                <x-form.button-group 
                                    cancelUrl="{{ route('training.sessions.index') }}"
                                    submitText="Schedule Session"
                                    submitIcon="bi-calendar-plus" />
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const startDatetime = document.getElementById('start_datetime');
    const endDatetime = document.getElementById('end_datetime');
    const programSelect = document.getElementById('training_program_id');

    // Auto-calculate end time based on program duration when start time changes
    startDatetime.addEventListener('change', function() {
        if (this.value && programSelect.value) {
            // This would require AJAX to get program duration
            // For now, just set a default 2-hour duration
            const startDate = new Date(this.value);
            const endDate = new Date(startDate.getTime() + (2 * 60 * 60 * 1000)); // Add 2 hours
            
            const year = endDate.getFullYear();
            const month = String(endDate.getMonth() + 1).padStart(2, '0');
            const day = String(endDate.getDate()).padStart(2, '0');
            const hours = String(endDate.getHours()).padStart(2, '0');
            const minutes = String(endDate.getMinutes()).padStart(2, '0');
            
            endDatetime.value = `${year}-${month}-${day}T${hours}:${minutes}`;
        }
    });

    // Validate end time is after start time
    endDatetime.addEventListener('change', function() {
        if (startDatetime.value && this.value) {
            if (new Date(this.value) <= new Date(startDatetime.value)) {
                alert('End time must be after start time');
                this.value = '';
            }
        }
    });
});
</script>
@endpush
@endsection
