<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Organization;
use App\Models\Employee;
use App\Models\PayrollSettings;
use App\Models\ApplicationMessage;
use App\Models\AuditLog;
use App\Models\PayrollApproval;
use App\Models\LoanRepayment;
use App\Models\TrainingEnrollment;
use App\Models\TrainingRecommendation;
use App\Models\PerformanceGoalProgress;
use App\Models\PerformanceCompetencyRating;
use App\Models\TrainingPerformanceImpact;
use App\Models\ApplicationStatusHistory;
use App\Models\PayrollApprovalWorkflow;
use App\Models\LearningPathEnrollment;
use App\Models\LearningPathStep;
use App\Models\InterviewQuestion;
use App\Models\TrainingMaterial;
use Illuminate\Foundation\Testing\RefreshDatabase;

class MultiTenancySecurityTest extends TestCase
{
    use RefreshDatabase;

    protected $org1;
    protected $org2;
    protected $user1;
    protected $user2;

    protected function setUp(): void
    {
        parent::setUp();

        // Create two organizations
        $this->org1 = Organization::factory()->create(['name' => 'Organization 1']);
        $this->org2 = Organization::factory()->create(['name' => 'Organization 2']);

        // Create users for each organization
        $this->user1 = User::factory()->create([
            'organization_id' => $this->org1->id,
            'email' => 'user1@org1.com'
        ]);

        $this->user2 = User::factory()->create([
            'organization_id' => $this->org2->id,
            'email' => 'user2@org2.com'
        ]);
    }

    /** @test */
    public function payroll_settings_are_isolated_by_organization()
    {
        $settings1 = PayrollSettings::factory()->create(['organization_id' => $this->org1->id]);
        $settings2 = PayrollSettings::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = PayrollSettings::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
        $this->assertNotContains($settings2->id, $results->pluck('id'));
    }

    /** @test */
    public function application_messages_are_isolated_by_organization()
    {
        $message1 = ApplicationMessage::factory()->create(['organization_id' => $this->org1->id]);
        $message2 = ApplicationMessage::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = ApplicationMessage::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function audit_logs_are_isolated_by_organization()
    {
        $log1 = AuditLog::factory()->create(['organization_id' => $this->org1->id]);
        $log2 = AuditLog::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = AuditLog::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function payroll_approvals_are_isolated_by_organization()
    {
        $approval1 = PayrollApproval::factory()->create(['organization_id' => $this->org1->id]);
        $approval2 = PayrollApproval::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = PayrollApproval::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function loan_repayments_are_isolated_by_organization()
    {
        $repayment1 = LoanRepayment::factory()->create(['organization_id' => $this->org1->id]);
        $repayment2 = LoanRepayment::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = LoanRepayment::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function training_enrollments_are_isolated_by_organization()
    {
        $enrollment1 = TrainingEnrollment::factory()->create(['organization_id' => $this->org1->id]);
        $enrollment2 = TrainingEnrollment::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = TrainingEnrollment::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function training_recommendations_are_isolated_by_organization()
    {
        $rec1 = TrainingRecommendation::factory()->create(['organization_id' => $this->org1->id]);
        $rec2 = TrainingRecommendation::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = TrainingRecommendation::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function performance_goal_progress_is_isolated_by_organization()
    {
        $progress1 = PerformanceGoalProgress::factory()->create(['organization_id' => $this->org1->id]);
        $progress2 = PerformanceGoalProgress::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = PerformanceGoalProgress::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function performance_competency_ratings_are_isolated_by_organization()
    {
        $rating1 = PerformanceCompetencyRating::factory()->create(['organization_id' => $this->org1->id]);
        $rating2 = PerformanceCompetencyRating::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = PerformanceCompetencyRating::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function training_performance_impacts_are_isolated_by_organization()
    {
        $impact1 = TrainingPerformanceImpact::factory()->create(['organization_id' => $this->org1->id]);
        $impact2 = TrainingPerformanceImpact::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = TrainingPerformanceImpact::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function application_status_histories_are_isolated_by_organization()
    {
        $history1 = ApplicationStatusHistory::factory()->create(['organization_id' => $this->org1->id]);
        $history2 = ApplicationStatusHistory::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = ApplicationStatusHistory::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function payroll_approval_workflows_are_isolated_by_organization()
    {
        $workflow1 = PayrollApprovalWorkflow::factory()->create(['organization_id' => $this->org1->id]);
        $workflow2 = PayrollApprovalWorkflow::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = PayrollApprovalWorkflow::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function learning_path_enrollments_are_isolated_by_organization()
    {
        $enrollment1 = LearningPathEnrollment::factory()->create(['organization_id' => $this->org1->id]);
        $enrollment2 = LearningPathEnrollment::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = LearningPathEnrollment::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function learning_path_steps_are_isolated_by_organization()
    {
        $step1 = LearningPathStep::factory()->create(['organization_id' => $this->org1->id]);
        $step2 = LearningPathStep::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = LearningPathStep::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function interview_questions_are_isolated_by_organization()
    {
        $question1 = InterviewQuestion::factory()->create(['organization_id' => $this->org1->id]);
        $question2 = InterviewQuestion::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = InterviewQuestion::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function training_materials_are_isolated_by_organization()
    {
        $material1 = TrainingMaterial::factory()->create(['organization_id' => $this->org1->id]);
        $material2 = TrainingMaterial::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);
        $results = TrainingMaterial::all();

        $this->assertCount(1, $results);
        $this->assertEquals($this->org1->id, $results->first()->organization_id);
    }

    /** @test */
    public function organization_id_is_automatically_set_on_create()
    {
        $this->actingAs($this->user1);

        $settings = PayrollSettings::create([
            'paye_bands' => PayrollSettings::getDefaultPayeBands(),
            'config_version' => '2026.02',
        ]);

        $this->assertEquals($this->org1->id, $settings->organization_id);
    }

    /** @test */
    public function cannot_access_other_organization_data_directly()
    {
        $settings2 = PayrollSettings::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);

        // Try to find org2's settings - should return null due to global scope
        $result = PayrollSettings::find($settings2->id);

        $this->assertNull($result);
    }

    /** @test */
    public function can_bypass_scope_with_withoutOrganizationScope()
    {
        $settings1 = PayrollSettings::factory()->create(['organization_id' => $this->org1->id]);
        $settings2 = PayrollSettings::factory()->create(['organization_id' => $this->org2->id]);

        $this->actingAs($this->user1);

        // With scope - should only see org1
        $withScope = PayrollSettings::all();
        $this->assertCount(1, $withScope);

        // Without scope - should see both (for admin purposes)
        $withoutScope = PayrollSettings::withoutOrganizationScope()->get();
        $this->assertCount(2, $withoutScope);
    }
}
