<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;
use App\Models\User;
use App\Models\Organization;
use App\Models\Department;
use App\Models\JobPosting;
use App\Models\Application;

class RecruitmentModuleTest extends TestCase
{
    use RefreshDatabase, WithFaker;

    protected $user;
    protected $organization;
    protected $department;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test organization
        $this->organization = Organization::factory()->create();
        
        // Create test user
        $this->user = User::factory()->create([
            'organization_id' => $this->organization->id
        ]);
        
        // Create test department
        $this->department = Department::factory()->create([
            'organization_id' => $this->organization->id
        ]);
        
        // Set current organization context
        session(['current_organization_id' => $this->organization->id]);
    }

    /** @test */
    public function authenticated_user_can_view_job_postings_index()
    {
        $this->actingAs($this->user);
        
        $response = $this->get(route('job-postings.index'));
        
        $response->assertStatus(200);
        $response->assertViewIs('recruitment.job-postings.index');
    }

    /** @test */
    public function authenticated_user_can_create_job_posting()
    {
        $this->actingAs($this->user);
        
        $jobData = [
            'title' => 'Software Developer',
            'department_id' => $this->department->id,
            'location' => 'Nairobi, Kenya',
            'employment_type' => 'full-time',
            'description' => 'We are looking for a talented software developer...',
            'requirements' => 'Bachelor\'s degree in Computer Science...',
            'responsibilities' => 'Develop and maintain web applications...',
            'status' => 'active',
        ];
        
        $response = $this->post(route('job-postings.store'), $jobData);
        
        $response->assertRedirect(route('job-postings.index'));
        $this->assertDatabaseHas('job_postings', [
            'title' => 'Software Developer',
            'organization_id' => $this->organization->id,
        ]);
    }

    /** @test */
    public function public_can_view_job_application_form()
    {
        $jobPosting = JobPosting::factory()->create([
            'organization_id' => $this->organization->id,
            'department_id' => $this->department->id,
            'status' => 'active',
        ]);
        
        $response = $this->get(route('public.apply.show', $jobPosting->public_token));
        
        $response->assertStatus(200);
        $response->assertViewIs('recruitment.public.apply');
        $response->assertSee($jobPosting->title);
    }

    /** @test */
    public function public_can_submit_job_application()
    {
        Storage::fake('public');
        
        $jobPosting = JobPosting::factory()->create([
            'organization_id' => $this->organization->id,
            'department_id' => $this->department->id,
            'status' => 'active',
        ]);
        
        $cvFile = UploadedFile::fake()->create('resume.pdf', 1000, 'application/pdf');
        
        $applicationData = [
            'first_name' => 'John',
            'last_name' => 'Doe',
            'email' => 'john.doe@example.com',
            'phone' => '0712345678',
            'cv' => $cvFile,
            'cover_letter' => 'I am very interested in this position...',
        ];
        
        $response = $this->post(route('public.apply.store', $jobPosting->public_token), $applicationData);
        
        $response->assertStatus(200);
        $response->assertViewIs('recruitment.public.success');
        
        $this->assertDatabaseHas('applications', [
            'job_posting_id' => $jobPosting->id,
            'email' => 'john.doe@example.com',
            'organization_id' => $this->organization->id,
        ]);
    }

    /** @test */
    public function duplicate_application_is_handled_properly()
    {
        Storage::fake('public');
        
        $jobPosting = JobPosting::factory()->create([
            'organization_id' => $this->organization->id,
            'department_id' => $this->department->id,
            'status' => 'active',
        ]);
        
        // Create existing application
        Application::factory()->create([
            'job_posting_id' => $jobPosting->id,
            'email' => 'john.doe@example.com',
            'organization_id' => $this->organization->id,
            'status' => 'new',
        ]);
        
        $cvFile = UploadedFile::fake()->create('resume.pdf', 1000, 'application/pdf');
        
        $applicationData = [
            'first_name' => 'John',
            'last_name' => 'Doe',
            'email' => 'john.doe@example.com',
            'phone' => '0712345678',
            'cv' => $cvFile,
            'cover_letter' => 'I am very interested in this position...',
        ];
        
        $response = $this->post(route('public.apply.store', $jobPosting->public_token), $applicationData);
        
        $response->assertStatus(200);
        $response->assertViewIs('recruitment.public.update-application');
    }

    /** @test */
    public function authenticated_user_can_view_recruitment_reports()
    {
        $this->actingAs($this->user);
        
        // Create some test data
        $jobPosting = JobPosting::factory()->create([
            'organization_id' => $this->organization->id,
            'department_id' => $this->department->id,
        ]);
        
        Application::factory()->count(5)->create([
            'job_posting_id' => $jobPosting->id,
            'organization_id' => $this->organization->id,
        ]);
        
        $response = $this->get(route('recruitment.reports.index'));
        
        $response->assertStatus(200);
        $response->assertViewIs('recruitment.reports.index');
    }

    /** @test */
    public function job_posting_generates_unique_public_token()
    {
        $jobPosting1 = JobPosting::factory()->create([
            'organization_id' => $this->organization->id,
            'department_id' => $this->department->id,
        ]);
        
        $jobPosting2 = JobPosting::factory()->create([
            'organization_id' => $this->organization->id,
            'department_id' => $this->department->id,
        ]);
        
        $this->assertNotEquals($jobPosting1->public_token, $jobPosting2->public_token);
        $this->assertNotEmpty($jobPosting1->public_token);
        $this->assertNotEmpty($jobPosting2->public_token);
    }

    /** @test */
    public function application_creates_status_history_on_creation()
    {
        $jobPosting = JobPosting::factory()->create([
            'organization_id' => $this->organization->id,
            'department_id' => $this->department->id,
        ]);
        
        $application = Application::factory()->create([
            'job_posting_id' => $jobPosting->id,
            'organization_id' => $this->organization->id,
        ]);
        
        $this->assertDatabaseHas('application_status_histories', [
            'application_id' => $application->id,
            'to_status' => 'new',
            'from_status' => null,
        ]);
    }

    /** @test */
    public function application_status_change_creates_history_record()
    {
        $this->actingAs($this->user);
        
        $jobPosting = JobPosting::factory()->create([
            'organization_id' => $this->organization->id,
            'department_id' => $this->department->id,
        ]);
        
        $application = Application::factory()->create([
            'job_posting_id' => $jobPosting->id,
            'organization_id' => $this->organization->id,
            'status' => 'new',
        ]);
        
        // Update status
        $application->update(['status' => 'screening']);
        
        $this->assertDatabaseHas('application_status_histories', [
            'application_id' => $application->id,
            'from_status' => 'new',
            'to_status' => 'screening',
        ]);
    }
}